(ns burningswell.worker.import
  (:require [burningswell.config.core :as config]
            [burningswell.db.countries :as countries]
            [burningswell.db.regions :as regions]
            [burningswell.db.spots :as spots]
            [burningswell.rabbitmq.core :as rmq]
            [burningswell.worker.subscriber :refer [subscriber]]
            [clojure.edn :as edn]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [commandline.core :refer [print-help with-commandline]]
            [environ.core :refer [env]]
            [schema.core :as s])
  (:import burningswell.worker.subscriber.Subscriber
           com.rabbitmq.client.Channel))

(defn full-name [spot region country]
  (->> [(:name spot) (:name region) (:name country)]
       (remove str/blank?)
       (str/join ", ")))

(s/defn ^:always-validate import-spot
  "Import the `spot`."
  [{:keys [db]} :- Subscriber spot :- s/Any]
  (when-let [location (:location spot)]
    (let [country (countries/closest db location)
          region (regions/closest db location)
          spot-name (full-name spot region country)]
      (if (empty? (spots/within-distance db location 0.3))
        (let [spot (spots/insert db {:_embedded
                                     {:country country
                                      :region region}
                                     :name (:name spot)
                                     :location location})]
          (log/infof "%s imported." spot-name)
          spot)
        (log/infof "%s not imported, it already exists." spot-name)))))

(s/defn ^:always-validate on-spot
  "Import the `spot`."
  [worker :- Subscriber channel :- Channel metadata :- s/Any spot :- s/Any]
  (when-let [spot (import-spot worker spot)]
    (rmq/publish channel "api" "spots.created" spot)
    (log/info {:msg "Spot imported" :spot spot})))

(def subscriptions
  "The subscriptions of the import worker."
  [{:name ::import-spot
    :exchange {:name "import" :type :direct :durable true}
    :handler on-spot
    :routing-keys ["spot"]
    :queue {:name "import.spot"
            :auto-delete false
            :durable true}
    :dead-letter true}])

(defn new-worker
  "Return a new import worker."
  [& [config]]
  (subscriber subscriptions config))

(defn import-spots
  "Import the spots from `input`."
  [config input]
  (rmq/with-rabbitmq [broker config]
    (rmq/with-channel [channel broker]
      (with-open [reader (io/reader input)]
        (doseq [line (line-seq reader)]
          (let [spot (edn/read-string {:readers *data-readers*} line)]
            (rmq/publish channel "import" "spot" spot)))))))

(defn -main [& args]
  (with-commandline [[opts input] args]
    [[h help "Print this help."]]
    (when (or (:help opts) (str/blank? input))
      (print-help "import FILE")
      (System/exit 1))
    (import-spots (config/broker env) input)
    (System/exit 0)))
