(ns burningswell.worker.flickr
  (:require [burningswell.db.photos :as photos]
            [burningswell.db.schemas :refer :all]
            [burningswell.db.spots :as spots]
            [burningswell.flickr :as flickr]
            [burningswell.rabbitmq.core :as rmq]
            [burningswell.worker.subscriber :refer [subscriber]]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [geo.core :as geo]
            [geo.postgis :refer [point]]
            [schema.core :as s])
  (:import burningswell.worker.subscriber.Subscriber
           com.rabbitmq.client.Channel))

(def commercial-licenses
  "Flickr licenses that allow photos to be used on commercial sites."
  (map :id (filter :usable? flickr/licenses)))

(s/defn ^:always-validate flickr-client
  "Return the Flickr client for worker."
  [worker :- Subscriber]
  (flickr/client (-> worker :flickr :api-key)))

(defn search-criterias
  "Return a vector of search criterias for `spot`."
  [spot]
  (let [country (-> spot :_embedded :country)
        license (str/join ","  commercial-licenses)
        location (:location spot)]
    [{:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (str "surf, " (:name spot) ", " (:name country))
      :radius 1
      :tags "surf"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (str "surfing, " (:name spot) ", " (:name country))
      :radius 1
      :tags "surf"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (str "surf, " (:name spot))
      :radius 1
      :tags "surf"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (str "surfing, " (:name spot))
      :radius 1
      :tags "surf"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (:name spot)
      :radius 1
      :tags "surf"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (:name spot)
      :radius 1
      :tags "surfing"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text (:name spot)
      :radius 1
      :tags "beach"
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :tags "surf"
      :radius 1
      :sort "relevance"}

     {:license license
      :lon (geo/point-x location)
      :lat (geo/point-y location)
      :text "surf"
      :radius 1
      :sort "relevance"}

     {:license license
      :text (:name spot)
      :tags "surf"
      :sort "relevance"}

     {:license license
      :text (:name spot)
      :tags "surfing"
      :sort "relevance"}

     {:license license
      :text (:name spot)
      :tags "beach"
      :sort "relevance"}

     {:license license
      :text (:name spot)
      :tags "city"
      :sort "relevance"}

     {:license license
      :text (:name spot)
      :sort "relevance"}]))

(defn log-spot [spot]
  (select-keys spot [:id :name]))

(defn log-photo [photo]
  (select-keys photo [:id :width :height]))

(defn make-photo [spot photo details size user]
  {:_embedded {:spot spot}
   :flickr
   {:id (Long/parseLong (:id photo))
    :owner
    {:id (:owner photo)
     :name (-> user :username :_content)
     :url (-> user :profileurl :_content)}}
   :location
   (if-let [location (:location details)]
     (point 4326
            (Double/parseDouble (:longitude location))
            (Double/parseDouble (:latitude location))))
   :title (:title photo)
   :url (:source size)})

(s/defn ^:always-validate search-photos
  "Handle created spots."
  [worker :- Subscriber spot :- Spot]
  (let [client (flickr-client worker)]
    (log/debug {:msg "Search spot photos" :spot (log-spot spot)})
    (for [criteria (search-criterias spot)
          photo (flickr/search client criteria)
          size (flickr/sizes client (:id photo))
          :when (= "original" (str/lower-case (:label size)))
          :let [user (flickr/user-info client (:owner photo))
                details (flickr/photo-info client (:id photo))]]
      (make-photo spot photo details size user))))

(s/defn ^:always-validate save-photo :- Photo
  "Save the Flickr `photo` of `spot`."
  [worker :- Subscriber spot :- Spot photo]
  (let [photo (photos/save (:db worker) photo)]
    (photos/add-spot (:db worker) photo spot)
    (log/debug {:msg "Save photo"
                :photo (log-photo photo)
                :spot (log-spot spot)})
    photo))

(s/defn ^:always-validate import-photo :- Photo
  "Import the Flickr `photo` of `spot`."
  [worker :- Subscriber spot :- Spot photo]
  (let [photo (save-photo worker spot photo)]
    (rmq/with-channel [channel (:broker worker)]
      (rmq/publish channel "api" "photos.created" photo)
      photo)))

(s/defn ^:always-validate import-photos :- [Photo]
  "Import Flickr photos for `spot`."
  [worker :- Subscriber spot :- Spot]
  (if-let [spot (spots/by-id (:db worker) (:id spot))]
    (let [photos (->> (search-photos worker spot)
                      (take 5)
                      (map #(import-photo worker spot %))
                      (doall))]
      (->> (assoc-in spot [:_embedded :photo] (first photos))
           (spots/update (:db worker)))
      (log/info {:msg (format "Saved %s Flickr photos for the spot %s."
                              (count photos) (:name spot))
                 :spot (log-spot spot)})
      photos)))

(s/defn ^:always-validate on-spot-created
  "Import Flickr photos for new spots."
  [worker :- Subscriber channel :- Channel metadata :- s/Any spot :- Spot]
  (import-photos worker spot))

(s/defn ^:always-validate on-spot-updated
  "Import Flickr photos for updated spots, that don't have any photos yet."
  [worker :- Subscriber channel :- Channel metadata :- s/Any spot :- Spot]
  (when (empty? (photos/by-spot (:db worker) spot))
    (import-photos worker spot)))

(def subscriptions
  [{:name ::spot-created
    :exchange {:name "api" :type :topic :durable true}
    :handler on-spot-created
    :routing-keys ["spots.created"]
    :queue {:name "flickr.spots.created"
            :auto-delete false
            :durable true}
    :dead-letter true}
   {:name ::spot-updated
    :exchange {:name "api" :type :topic :durable true}
    :handler on-spot-updated
    :routing-keys ["spots.updated"]
    :queue {:name "flickr.spots.updated"
            :auto-delete false
            :durable true}
    :dead-letter true}])

(defn new-worker
  "Return a new Flickr worker."
  [& [config]]
  (subscriber subscriptions config))
