(ns burningswell.worker.spot-weather
  (:require [burningswell.db.schemas :refer :all]
            [burningswell.db.spots :as spots]
            [burningswell.db.weather :as weather]
            [burningswell.worker.util :as util]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [schema.core :as s])
  (:import [com.rabbitmq.client Channel]))

(s/defrecord Worker
    [channel :- (s/maybe Channel)
     subscriptions :- [s/Any]]
  {s/Any s/Any})

(s/defn ^:always-validate on-spot-created
  "Create the weather current forecast for `spot`."
  [{:keys [db] :as worker} :- Worker
   channel :- Channel
   metadata :- s/Any
   spot :- Spot]
  (when-let [spot (spots/by-id db (:id spot))]
    (weather/import-spot-forecasts db {:spots [spot]})
    @(weather/refresh-3-hourly-spot-weather db)))

(extend-protocol component/Lifecycle
  Worker
  (start [worker]
    (util/start-subscriber worker))
  (stop [worker]
    (util/stop-subscriber worker)))

(def subscriptions
  [{:name ::spot-created
    :exchange {:name "api" :type :topic :durable true}
    :handler on-spot-created
    :routing-keys ["spots.created"]
    :queue {:name "weather.spots.created"
            :auto-delete false
            :durable true}
    :dead-letter true}])

(defn new-worker
  "Return a new Flickr worker."
  [& [config]]
  (component/using
   (map->Worker (assoc config :subscriptions subscriptions))
   [:broker :db]))
