(ns burningswell.worker.import
  (:require [burningswell.config.core :as config]
            [burningswell.db.countries :as countries]
            [burningswell.db.regions :as regions]
            [burningswell.db.schemas :refer :all]
            [burningswell.db.spots :as spots]
            [burningswell.rabbitmq.core :as rmq]
            [burningswell.worker.util :as util]
            [clojure.edn :as edn]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [commandline.core :refer [print-help with-commandline]]
            [com.stuartsierra.component :as component]
            [datumbazo.core :refer [select]]
            [environ.core :refer [env]]
            [schema.core :as s])
  (:import [com.rabbitmq.client Channel]))

(s/defrecord Worker
    [channel :- (s/maybe Channel)
     subscriptions :- [s/Any]]
  {s/Any s/Any})

(defn full-name [spot region country]
  (->> [(:name spot) (:name region) (:name country)]
       (remove str/blank?)
       (str/join ", ")))

(s/defn ^:always-validate import-spot
  "Import the `spot`."
  [{:keys [db]} :- Worker spot :- s/Any]
  (when-let [location (:location spot)]
    (let [country (countries/closest db location)
          region (regions/closest db location)
          spot-name (full-name spot region country)]
      (if (empty? (spots/within-distance db location 0.3))
        (let [spot (spots/insert db {:_embedded
                                     {:country country
                                      :region region}
                                     :name (:name spot)
                                     :location location})]
          (log/infof "%s imported." spot-name)
          spot)
        (log/infof "%s not imported, it already exists." spot-name)))))

(s/defn ^:always-validate on-spot
  "Import the `spot`."
  [worker :- Worker channel :- Channel metadata :- s/Any spot :- s/Any]
  (when-let [spot (import-spot worker spot)]
    (rmq/publish channel "api" "spots.created" spot)
    (log/info {:msg "Spot imported" :spot spot})))

(extend-protocol component/Lifecycle
  Worker
  (start [worker]
    (util/start-subscriber worker))
  (stop [worker]
    (util/stop-subscriber worker)))

(def subscriptions
  "The subscriptions of the import worker."
  [{:name ::import-spot
    :exchange {:name "import" :type :direct :durable true}
    :handler on-spot
    :routing-keys ["spot"]
    :queue {:name "import.spot"
            :auto-delete false
            :durable true}
    :dead-letter true}])

(defn new-worker
  "Return a new import worker."
  [& [config]]
  (component/using
   (map->Worker (assoc config :subscriptions subscriptions))
   [:broker :db]))

(defn import-spots
  "Import the spots from `input`."
  [config input]
  (rmq/with-rabbitmq [broker config]
    (rmq/with-channel [channel broker]
      (with-open [reader (io/reader input)]
        (doseq [line (line-seq reader)]
          (let [spot (edn/read-string {:readers *data-readers*} line)]
            (rmq/publish channel "import" "spot" spot)))))))

(defn -main [& args]
  (with-commandline [[input] args]
    [[h help "Print this help."]]
    (when (or help (str/blank? input))
      (print-help "import FILE")
      (System/exit 1))
    (import-spots (config/broker env) input)
    (System/exit 0)))
