(ns burningswell.web.ui.forms.signup
  (:require [apollo.core :as a]
            [burningswell.web.ui.buttons :as button]
            [burningswell.web.ui.links :as links]
            [burningswell.web.ui.social :as social]
            [burningswell.web.ui.text-field :as text-field]
            [clojure.pprint :refer [pprint]]
            [clojure.string :as str]
            [sablono.core :refer [html]]))

(def default-data
  {:__typename "SignupForm"
   :state "ready"
   :id nil
   :email
   {:__typename "FormField"
    :errors []
    :value ""}
   :username
   {:__typename "FormField"
    :errors []
    :value ""}
   :password
   {:__typename "FormField"
    :errors []
    :value ""}})

(def fsm
  {"ready"          {:invalid-email    "email-error"
                     :invalid-username "username-error"
                     :invalid-password "password-error"
                     :submit           "submitting"}
   "email-error"    {:change-email     "ready"}
   "username-error" {:change-username  "ready"}
   "password-error" {:change-password  "ready"}
   "submit-error"   {:change-email     "ready"
                     :change-username  "ready"
                     :change-password  "ready"}
   "submitting"     {:submit-success   "ready"
                     :submit-error     "submit-error"}})

(a/defgql signup-query
  '((query
     Signup
     [($id String)]
     (signup
      [(id $id)]
      ((client))
      id
      (username value errors)
      (email value errors)
      (password value errors)
      state
      __typename))))

(a/defgql validate-mutation
  '((mutation
     Validate
     [($input SignupInput!)]
     (validate_signup
      [(input $input)]))))

(a/defgql submit-mutation
  '((mutation
     Signup
     [($input SignupInput!)]
     (signup
      [(input $input)]
      auth_token
      (user id username)))))

(defn- field-value
  "Returns the value of `field-name` from the form `data`."
  [data field-name]
  (get-in data [:signup (keyword field-name) :value]))

(defn- field-errors
  "Returns the errors of `field-name` from the form `data`."
  [data field-name]
  (get-in data [:signup (keyword field-name) :errors]))

(defn- field-error
  "Returns the first error of `field-name` from the form `data`."
  [data field]
  (when-let [error (first (field-errors data field))]
    (str (-> field name str/capitalize) " " error ".")))

(defn- signup-variables
  "Returns the variables for the signup mutation."
  [data]
  {:input {:email (field-value data :email)
           :username (field-value data :username)
           :password (field-value data :password)}})

(defn- read-state [client form]
  (a/read-query client signup-query {:variables {:id (:id form)}}))

(defn- update-state! [client form f & args]
  (let [state (read-state client form)]
    (a/write-data! client (apply f state args))))

(defn- update-errors! [client form field errors]
  (update-state! client form assoc-in [:signup (keyword field) :errors] (seq errors)))

(defn- update-value! [client form field value]
  (update-state! client form assoc-in [:signup (keyword field) :value] value))

(defn- transition!
  "Updates app-state to contain the state reached by transitioning from the
  current state."
  [client form transition]
  (let [current-state (-> (read-state client form) :signup :state)
        next-state (get-in fsm [current-state transition])]
    ;; (prn "TRANSITION:" current-state "->" transition "->" next-state)
    (if next-state
      (update-state! client form assoc-in [:signup :state] next-state))))

(defn- resolve-data [obj args context info]
  (if-let [id (:id args)]
    (assoc default-data :id id)
    default-data))

(defn- submit-form [client form event]
  ;; (pprint (read-data client form))
  (.preventDefault event))

(defn- value
  "Returns the value of the `event` target."
  [event]
  (.. event -target -value))

(defn- error-messages [result]
  (-> (get-in result [:networkError :result :errors])
      first :context :messages))

(defn- error-messages-by-field [result field]
  (seq (get-in (error-messages result) [:input (keyword field)])))

(defn- validate-field! [client form field validate]
  (let [data (read-state client form)]
    (-> (validate {:variables (signup-variables data)})
        (.then (fn [{:keys [data]}]
                 (update-errors! client form  field [])
                 (transition! client form (keyword (str "change-" (name field))))))
        (.catch (fn [result]
                  (if-let [errors (error-messages-by-field result field)]
                    (do (update-errors! client form field errors)
                        (transition! client form (keyword (str "invalid-" (name field)))))
                    (do (update-errors! client form field [])
                        (transition! client form (keyword (str "change-" (name field)))))))))))

(defn- on-email-change
  "Handle an email change `event`."
  [client form validate event]
  (update-value! client form :email (value event))
  (validate-field! client form :email validate))

(defn- on-username-change
  "Handle an username change `event`."
  [client form validate event]
  (update-value! client form :username (value event))
  (validate-field! client form :username validate))

(defn- on-password-change
  "Handle a password change `event`."
  [client form validate event]
  (update-value! client form :password (value event))
  (validate-field! client form :password validate))

(defn- assoc-errors [state field errors]
  (assoc-in state [:signup field :errors] errors))

(defn- assoc-value [state field value]
  (assoc-in state [:signup field :value] value))

(defn- reset-field [state field]
  (-> (assoc-errors state field [])
      (assoc-value field "")))

(defn- clear-form! [client form]
  (update-state! client form
                 #(-> (reset-field % :email)
                      (reset-field :password)
                      (reset-field :username))))

(defn- on-submit-success
  [client form data]
  (clear-form! client form)
  (transition! client form :submit-success)
  (when-let [handler (:on-signup form)]
    (handler data)))

(defn- on-submit-error [client form data]
  (prn "FAILURE")
  (transition! client form :submit-error))

(defn- on-submit
  "Handle a submit `event`."
  [client form action event]
  (.preventDefault event)
  (transition! client form :submit)
  (-> (action)
      (.then (fn [{:keys [data]}]
               (if-let [token (-> data :signup :auth_token)]
                 (on-submit-success client form data)
                 (on-submit-error client form data))))
      (.catch (fn [result]
                (doseq [field [:email :username :password]]
                  (->> (error-messages-by-field result field)
                       (update-errors! client form field)))
                (transition! client form :submit-error)))))

(defn- email-field
  "Render the email field."
  [form data]
  (let [error (field-error data :email)]
    (a/with-mutation [validate {:keys [client]}]
      {:mutation validate-mutation}
      (text-field/text-field
       {:auto-complete "email"
        :label "Email"
        :helper-text
        {:persistent? (some? error)
         :text (or error "Enter your email.")
         :valid? (nil? error)
         :validation-message? (some? error)
         :validation? (some? error)}
        :on-change #(on-email-change client form validate %)
        :type "email"
        :value (-> data :signup :email :value)}))))

(defn- username-field
  "Render the username field."
  [form data]
  (let [error (field-error data :username)]
    (a/with-mutation [validate {:keys [client]}]
      {:mutation validate-mutation}
      (text-field/text-field
       {:label "Username"
        :helper-text
        {:persistent? (some? error)
         :text (or error "Enter your username.")
         :valid? (nil? error)
         :validation-message? (some? error)
         :validation? (some? error)}
        :on-change #(on-username-change client form validate %)
        :value (-> data :signup :username :value)}))))

(defn- password-field
  "Render the password field."
  [form data]
  (let [error (field-error data :password)]
    (a/with-mutation [validate {:keys [client]}]
      {:mutation validate-mutation}
      (text-field/text-field
       {:label "Password"
        :helper-text
        {:persistent? (some? error)
         :text (or error "Choose a safe password.")
         :valid? (nil? error)
         :validation-message? (some? error)
         :validation? (some? error)}
        :type "password"
        :on-change #(on-password-change client form validate %)
        :value (-> data :signup :password :value)}))))

(defn- submit-button
  "Render the submit button."
  [form data]
  (a/with-mutation [action {:keys [client]}]
    {:mutation submit-mutation
     :variables (signup-variables data)}
    (button/button
     "Signup"
     {:disabled (not= (-> data :signup :state) "ready")
      :raised true
      :on-click #(on-submit client form action %)})))

(defn signup-form [form]
  (a/with-query [{:keys [client data error loading]}]
    {:query signup-query :variables {:id (:id form)}}
    (when (:signup data)
      (html
       [:form.signup-form
        [:div.signup-form__username
         (username-field form data)]
        [:div.signup-form__email
         (email-field form data)]
        [:div.signup-form__password
         (password-field form data)]
        [:div.signup-form__submit
         (submit-button form data)]
        [:div.signup-form__terms
         "By clicking \"Signup\", you agree to our "
         (links/terms-of-service) " and " (links/privacy-statement) ". "
         "We’ll occasionally send you account related emails."]
        [:div.signup-form__connect
         (social/connect)]
        [:div.signup-form__signin
         [:div "Already have an account? "]
         (links/signin)]]))))
