(ns burningswell.web.ui.forms.reset-password
  (:require [apollo.core :as a]
            [burningswell.web.ui.buttons :as button]
            [burningswell.web.ui.links :as links]
            [burningswell.web.ui.social :as social]
            [burningswell.web.ui.text-field :as text-field]
            [clojure.string :as str]
            [sablono.core :refer [html]]))

(def default-data
  {:__typename "ResetPasswordForm"
   :state "ready"
   :id nil
   :email
   {:__typename "FormField"
    :errors []
    :value ""}})

(def fsm
  {"ready"          {:missing-email    "email-error"
                     :submit           "submitting"}
   "email-error"    {:change-email     "ready"}
   "submit-error"   {:change-email     "ready"}
   "submitting"     {:submit-success   "ready"
                     :submit-error     "submit-error"}})

(a/defgql reset-password-query
  '((query
     ResetPassword
     [($id String)]
     (reset-password
      [(id $id)]
      ((client))
      id
      (email value errors)
      state
      __typename))))

(a/defgql reset-password-mutation
  '((mutation
     ResetPassword
     [($input ResetPasswordInput!)]
     (reset-password
      [(input $input)]
      (email id address)))))

(defn- read-state [client form]
  (a/read-query client reset-password-query {:variables {:id (:id form)}}))

(defn- update-state! [client form f & args]
  (let [state (read-state client form)]
    (a/write-data! client (apply f state args))))

(defn- update-errors! [client form field errors]
  (update-state! client form assoc-in [:reset_password (keyword field) :errors] (seq errors)))

(defn- resolve-data [obj args context info]
  (if-let [id (:id args)]
    (assoc default-data :id id)
    default-data))

;; State machine

(defn- transition!
  "Updates app-state to contain the state reached by transitioning from the
  current state."
  [client form transition]
  (let [current-state (-> (read-state client form) :reset_password :state)
        next-state (get-in fsm [current-state transition])]
    ;; (prn "TRANSITION:" current-state "->" transition "->" next-state)
    (if next-state
      (update-state! client form assoc-in [:reset_password :state] next-state))))

(defn- on-email-change
  [client form value]
  (let [app-state (update-state! client form assoc-in [:reset_password :email :value] value)]
    (if (str/blank? value)
      (let [errors ["Email can't be blank."]]
        (update-state! client form assoc-in [:reset_password :email :errors] errors)
        (transition! client form :missing-email))
      (transition! client form :change-email))))

(defn- assoc-errors [state field errors]
  (assoc-in state [:reset_password field :errors] errors))

(defn- assoc-value [state field value]
  (assoc-in state [:reset_password field :value] value))

(defn- reset-field [state field]
  (-> (assoc-errors state field [])
      (assoc-value field "")))

(defn- clear-form! [client form]
  (update-state! client form #(reset-field % :email)))

(defn- on-submit-success
  [client form data]
  (clear-form! client form)
  (transition! client form :submit-success)
  (when-let [handler (:on-reset-password form)]
    (handler data)))

(defn- on-submit-error [client form data]
  (prn "FAILURE")
  (transition! client form :submit-error))

(defn- error-messages [result]
  (-> (get-in result [:networkError :result :errors])
      first :context :messages))

(defn- error-messages-by-field [result field]
  (seq (get-in (error-messages result) [:input (keyword field)])))

(defn- on-submit
  [client form action]
  (transition! client form :submit)
  (-> (action)
      (.then (fn [{:keys [data]}]
               (if-let [token (-> data :reset_password :email :id)]
                 (on-submit-success client form data)
                 (on-submit-error client form data))))
      ;; (.catch #(prn "TODO: Handle backend error"))
      (.catch (fn [result]
                (prn "ERROR" (error-messages-by-field result :email))
                (->> (error-messages-by-field result :email)
                     (update-errors! client form :email))
                (transition! client form :submit-error)))))

(defn- field-errors
  "Returns the errors of `field-name` from the form `data`."
  [data field-name]
  (get-in data [:reset_password (keyword field-name) :errors]))

(defn- field-error
  "Returns the first error of `field-name` from the form `data`."
  [data field]
  (when-let [error (first (field-errors data field))]
    (str (-> field name str/capitalize) " " error ".")))

(defn- field-value
  "Returns the value of `field-name` from the form `data`."
  [data field-name]
  (get-in data [:reset_password (keyword field-name) :value]))

(defn- form-state
  "Returns the state of the FSM from `data`."
  [data]
  (-> data :reset_password :state))

(defn- value
  "Returns the value of the `event` target."
  [event]
  (.. event -target -value))

(defn- email-error
  "Returns the email error message."
  [data]
  (case (form-state data)
    "email-error" (-> data :reset_password :email :errors first)
    nil))

(defn- email-field
  "Render a email text field."
  [client form data]
  (let [error (field-error data :email)]
    (text-field/text-field
     {:helper-text
      {:persistent? (some? error)
       :text (or error "Please enter your email.")
       :valid? (nil? error)
       :validation-message? (some? error)
       :validation? (some? error)}
      :auto-complete "email"
      :label "Your email address"
      :type "email"
      :on-change #(on-email-change client form (value %))
      :value (field-value data :email)})))

(defn- reset-password-variables
  "Returns the variables for the sign in mutation."
  [data]
  {:input {:email (field-value data :email)}})

(defn- submit-button
  "Render a submit button."
  [form data]
  (a/with-mutation [action {:keys [client]}]
    {:mutation reset-password-mutation
     :variables (reset-password-variables data)}
    (button/button
     "Send reset instructions"
     {:disabled (not= (form-state data) "ready")
      :raised true
      :on-click #(do (on-submit client form action)
                     (.preventDefault %))})))

(defn form
  "Render a reset password form."
  [form]
  (a/with-query [{:keys [client data loading]}]
    {:query reset-password-query :variables {:id (:id form)}}
    (when-not loading
      (html [:form.reset-password-form
             [:div.reset-password-form__email
              (email-field client form data)]
             [:div.reset-password-form__submit
              (submit-button form data)]
             [:div.reset-password-form__connect
              (social/connect)]
             [:div.reset-password-form__signin
              "Did you remember? "
              (links/signin)]]))))
