(ns burningswell.web.ui.drawer
  (:refer-clojure :exclude [list])
  (:require ["@material/react-drawer" :default Drawer :refer [DrawerAppContent DrawerContent DrawerHeader DrawerTitle]]
            ["@material/react-list" :default List :refer [ListItem ListItemGraphic ListItemText]]
            [apollo.core :as a]
            [burningswell.web.authentication :as auth]
            [burningswell.web.core :as core]
            [burningswell.web.history :as history]
            [burningswell.web.location :as location]
            [burningswell.web.router :as router]
            [burningswell.web.ui.app-bar :as app-bar]
            [burningswell.web.ui.list :as list]
            [burningswell.web.ui.system :as system]
            [burningswell.web.ui.viewer :as viewer]
            [react :as React :refer [createElement]]
            [sablono.core :refer [html]]))

(defn- read-state [client]
  (a/read-query client (a/ast
                        '((drawer
                           ((client))
                           open)))))

(defn toggle! [client]
  (let [state (read-state client)]
    (->> (update-in state [:drawer :open] not)
         (a/write-data! client))))

(defn navigation-icon [client]
  (core/icon :menu {:onClick #(toggle! client)}))

(defn- make-props [opts children]
  (clj->js (assoc opts :children (or children []))))

(defn- create-title [opts name]
  (createElement DrawerTitle (make-props opts [name])))

(defn- create-header [opts & children]
  (createElement DrawerHeader (make-props opts children)))

(defn- create-content [opts & children]
  (createElement DrawerContent (make-props opts children)))

(defn- create-container [opts & children]
  (createElement Drawer (make-props opts children)))

(defn- create-app-content [opts & children]
  (createElement DrawerAppContent (make-props opts children)))

(defn- item [{:keys [icon key primary-text token]}]
  (system/consumer
   (fn [{:keys [client history]}]
     (list/item
      {:key key
       :onClick #(do (history/set-token! history token)
                     (toggle! client))}
      (list/item-graphic
       {:graphic (list/icon icon)})
      (list/item-text
       {:primaryText primary-text})))))

(defn- continents []
  (item {:key "drawer-continents"
         :icon :map
         :primary-text "Continents"
         :token "continents"}))

(defn- home []
  (item {:key "drawer-home"
         :icon :home
         :primary-text "Home"
         :token ""}))

(defn- countries []
  (item {:key "drawer-countries"
         :icon :terrain
         :primary-text "Countries"
         :token "countries"}))

(defn- regions []
  (item {:key "drawer-regions"
         :icon :account_balance
         :primary-text "Regions"
         :token "regions"}))

(defn- show-map []
  (item {:key "drawer-show-map"
         :icon :map
         :primary-text "Map"
         :token "map"}))

(defn- spots []
  (item {:key "drawer-spots"
         :icon :place
         :primary-text "Spots"
         :token "spots"}))

(defn- signin []
  (item {:key "drawer-signin"
         :icon :person
         :primary-text "Signin"
         :token "signin"}))

(defn- signup []
  (item {:key "drawer-signup"
         :icon :person_add
         :primary-text "Signup"
         :token "signup"}))

(defn- signout!
  "Signout the current user."
  [{:keys [authentication client location router]}]
  ;; TODO: Fix store reset while query was in flight(not completed in link chain)
  (auth/clear-auth-token! authentication)
  (router/match-current-url! router)
  (.then (a/reset-store! client)
         #(location/reset-store! location)))

(defn- signout []
  (system/consumer
   (fn [system]
     (list/item
      {:key "drawer-signout"
       :onClick #(signout! system)}
      (list/item-graphic
       {:graphic (list/icon :exit_to_app)})
      (list/item-text
       {:primaryText "Signout"})))))

(defn- users []
  (item {:key "drawer-users"
         :icon :wc
         :primary-text "Users"
         :token "users"}))

(defn- has-role?
  "Returns true if the `user` has the `role`, otherwise false."
  [user role]
  (contains? (set (map :name  (a/nodes user :roles))) role))

(defn- content-list []
  (a/with-query [{:keys [data]}]
    {:key "drawer-content-list"
     :query (a/ast '((viewer (user id (roles (edges (node id name)))))))}
    (let [logged-in? (-> data :viewer :user :id)
          admin? (has-role? (-> data :viewer :user) "admin")]
      (list/list
       {:singleSelection true
        :selectedIndex 1}
       (home)
       (continents)
       (countries)
       (regions)
       (spots)
       (show-map)
       (when admin? (users))
       (list/divider)
       (when logged-in? (signout))
       (when-not logged-in? (signin))
       (when-not logged-in? (signup))))))

(defn header [& [{:keys [title]}]]
  (create-header
   {:key "drawer-header"}
   (create-title
    {:key "drawer-title"}
    (or title "Burning Swell"))))

(defn content []
  (create-content
   {:key "drawer-content"}
   (content-list)))

(defn app-content [& children]
  (create-app-content
   {:className "drawer-app-content"
    :key "drawer-app-content"}
   children))

(defn drawer [& [{:keys [dismissible? modal?]}]]
  (a/query
   {:key "drawer"
    :query (a/ast '((drawer ((client)) open)))}
   (fn [{:keys [client data error loading] :as props}]
     (create-container
      {:dismissible dismissible?
       :modal modal?
       :onClose #(toggle! client)
       :open (true? (a/get data :drawer :open))}
      (header)
      (content)))))
