(ns burningswell.web.location
  (:require [apollo.core :as a]
            [burningswell.web.cookies :as cookies]
            [burningswell.web.logging :as log]
            [burningswell.web.storage :as storage]
            [com.stuartsierra.component :as component]))

(def logger
  "The logger of the current namespace."
  (log/logger (namespace ::logger)))

(a/defgql location-query
  '((location ((client)) latitude longitude)))

(defn make-location
  [latitude longitude]
  {:__typename "Location"
   :latitude latitude
   :longitude longitude})

(defn default-data
  "The default location data."
  [& [{:keys [latitude longitude]}]]
  (let [latitude (or latitude (get (storage/local-storage) :latitude))
        longitude (or longitude (get (storage/local-storage) :longitude))]
    (when (and latitude longitude)
      (make-location latitude longitude))))

(defn- read-state
  "Reads the state from the Apollo store."
  [{:keys [client]}]
  (a/read-query client location-query))

(defn read-location
  "Reads the current location from the Apollo store."
  [service]
  (:location (read-state service)))

(defn- update-state!
  "Update the current state with `f`."
  [{:keys [client] :as service} f & args]
  (a/write-data! client (apply f (read-state service) args)))

(defn- write-location-to-cookies!
  "Write the `location` to the cookies."
  [service {:keys [latitude longitude] :as location}]
  (let [opts {:path "/" :secure false}]
    (cookies/set-cookie! :latitude latitude opts)
    (cookies/set-cookie! :longitude longitude opts)
    (log/info logger (str "Wrote location: " latitude ", " longitude " to cookies."))))

(defn- write-location-to-store!
  "Write the `location` to the Apollo store."
  [service {:keys [latitude longitude] :as location}]
  (update-state! service merge {:location location})
  (log/info logger (str "Wrote location: " latitude ", " longitude " to Apollo store.")))

(defn- write-location-to-local-storage!
  "Write the `location` to the Apollo store."
  [{:keys [local-storage]} {:keys [latitude longitude]}]
  (when local-storage
    (assoc! local-storage :latitude latitude)
    (assoc! local-storage :longitude longitude)
    (log/info logger (str "Wrote location: " latitude ", " longitude " to local storage."))))

(defn- save-location!
  "Write the `location` to the Apollo store and local storage."
  [service location]
  (reset! (:location service) location)
  (write-location-to-cookies! service location)
  (write-location-to-local-storage! service location)
  (write-location-to-store! service location))

(defn- error-message
  "Returns the location `error` message."
  [error]
  (str "Can't get geo location: " (.-message error) " " "(" (.-code error) ")."))

(defn- navigator-location
  "Returns the Geo Location object from the navigator or nil if it does
  not exist."
  []
  (when (exists? js/navigator)
    (.-geolocation js/navigator)))

(defn- position->location
  "Convert `position` into a geo point."
  [position]
  (let [coords (.-coords position)]
    (make-location (.-latitude coords) (.-longitude coords))))

(defn reset-store!
  "Read the current geo location and reset it in the Apollo store."
  [service]
  (when (navigator-location)
    (.getCurrentPosition
     (navigator-location)
     (fn [position]
       (save-location! service (position->location position)))
     (fn [error]
       (log/warning logger (error-message error)))
     #js {:enableHighAccuracy true})))

(defn start-watch!
  "Start watching for geo location changes."
  [service]
  (when (navigator-location)
    (.watchPosition
     (navigator-location)
     (fn [position]
       (save-location! service (position->location position)))
     (fn [error]
       (log/warning logger (error-message error)))
     #js {:enableHighAccuracy true})))

(defn stop-watch!
  "Stop watching for geo location changes."
  [{:keys [listener] :as service}]
  (when (and (navigator-location) listener)
    (.clearWatch (navigator-location) listener)))

(defrecord Location [client listener location]
  component/Lifecycle
  (start [service]
    (if (or (navigator-location) (not listener))
      (let [listener (start-watch! service)]
        (log/info logger "Geo location successfully started.")
        (assoc service :listener listener))
      service))

  (stop [service]
    (if listener
      (do (stop-watch! service)
          (log/info logger "Geo location successfully stopped.")
          (assoc service :listener nil))
      service)))

(defn location
  "Returns a new location service component."
  []
  (-> (map->Location {:location (atom {})})
      (component/using [:client :local-storage])))

(comment
  (read-location (:location burningswell.web.system/*system*))
  (reset-store! (:location burningswell.web.system/*system*))
  )
