(ns burningswell.web.ui.cards.spot
  (:require [apollo.core :as a]
            [burningswell.web.history :as history]
            [burningswell.web.ui.system :as system]
            [burningswell.web.ui.layout :as layout]
            [burningswell.web.ui.buttons :as buttons]
            [burningswell.web.ui.cards.core :as cards]
            [burningswell.web.ui.icon :as icon]))

(def spot-card-fragment
  '(fragment
    spot (on Spot)
    id name
    (country
     id name
     (slug path))
    (region
     id name
     (slug path))
    (photo
     id
     (images
      [(first 1)
       (sort WIDTH)]
      (edges
       (node id url))))
    (slug path)))

(a/defgql spot-query
  `((query
     SpotCard
     [($id ID!)]
     (spot
      [(id $id)]
      (... spot)))
    ~spot-card-fragment))

(defn photo-url [spot]
  (-> spot :photo :images :edges first :node :url))

(defn card [{:keys [variables]}]
  (system/with-consumer [{:keys [history]}]
    (a/query
     {:query spot-query
      :variables variables}
     (fn [{:keys [client data error loading] :as props}]
       (when-not loading
         (when-let [spot (:spot data)]
           (cards/create-card
            {:class "spot-card"}
            (cards/create-primary-content
             {:class "spot-card__primary-content"
              :on-click #(history/set-path! history (-> spot :slug :path))}
             (if-let [url (photo-url spot)]
               (cards/create-card-media
                {:image-url url :wide true})
               [:div {:style {:height "200px"}}])
             [:h1.mdc-typography--headline5 (:name spot)])
            (cards/create-card-actions
             {:class "spot-card__card-actions"}
             #_(cards/create-card-action-buttons
                {:class "spot-card__card-action-buttons"}
                (buttons/button "More")
                (buttons/button "Bookmark"))
             (cards/create-card-action-icons
              {:class "spot-card__card-action-icons"}
              (buttons/icon-button (icon/icon :favorite_border))
              (buttons/icon-button (icon/icon :share))
              (buttons/icon-button (icon/icon :more_vert)))))))))))

(a/defgql spots-query
  `((query
     Spots
     [($countries [ID!])
      ($regions [ID!])]
     (spots
      [(countries $countries)
       (regions $regions)]
      (edges
       (node
        id
        (... spot)))))
    ~spot-card-fragment))

(defn cards [& [{:keys [variables]}]]
  (a/query
   {:query spots-query
    :variables variables}
   (fn [{:keys [client data error loading] :as props}]
     (when-not loading
       (when-let [spots (a/nodes data :spots)]
         (layout/grid
          {}
          (apply layout/row {}
                 (for [spot (a/nodes data :spots)]
                   (layout/cell
                    {:columns 2
                     :key (str "spot-card-" (:id spot))}
                    (card {:variables {:id (:id spot)}}))))))))))
