(ns burningswell.web.history
  (:require [burningswell.web.logging :as log]
            [clojure.string :as str]
            [com.stuartsierra.component :as component]
            [goog.events :as events]
            [goog.history.EventType :as EventType]
            [no.en.core :refer [format-url]])
  (:import goog.history.Html5History))

(def logger
  "The logger of the current namespace."
  (log/logger (namespace ::logger)))

(defrecord History [instance listeners]
  component/Lifecycle
  (start [this]
    (let [instance (Html5History.)]
      (.setUseFragment instance false)
      (.setEnabled instance true)
      (log/info logger "HTML5 history started.")
      (assoc this :instance instance)))
  (stop [this]
    (some-> instance (.setEnabled false))
    (doseq [listener @listeners]
      (events/unlistenByKey listener))
    (reset! listeners [])
    (log/info logger "HTML5 history stopped.")
    (assoc this :instance nil)))

(defn history
  "Returns a HTML5 history component."
  [& [config]]
  (-> (assoc config :listeners (atom []))
      (map->History)
      (component/using [:logging])))

(defn listen!
  "Register `handler` to listen for navigation events."
  [history handler]
  (let [{:keys [instance listeners]} history
        listener (goog.events/listen instance EventType/NAVIGATE handler)]
    (swap! listeners conj listener)
    listener))

(defn unlisten!
  "Unregister the history event `listener`."
  [history listener]
  (events/unlistenByKey listener)
  (swap! (:listeners history) (fn [listeners] (remove #(= % listener) listeners)))
  listener)

(defn path->token [path]
  (some-> path (str/replace #"^/+" "")))

(defn strip-query-params
  "Strip the query params in the `url` string."
  [url]
  (if-let [index (str/index-of url "?")]
    (subs url 0 index) url))

(defn remove-query-params!
  "Remove the query params from the browser location."
  []
  (let [url js/window.location.href]
    (js/window.history.replaceState
     #js {} js/document.title
     (strip-query-params url))))

(defn set-token! [history token]
  (remove-query-params!)
  (.setToken (:instance history) token))

(defn set-path! [history path]
  (set-token! history (path->token path)))

(defn set-url! [history url]
  (some->> (format-url (select-keys url [:uri :query-params]))
           (set-path! history)))

(defn replace-token! [history token]
  (remove-query-params!)
  (.replaceToken (:instance history) token))

(defn replace-path! [history path]
  (replace-token! history (path->token path)))
