(ns burningswell.web.client
  (:require [apollo-cache-inmemory :refer [InMemoryCache]]
            [apollo-client :refer [ApolloClient]]
            [apollo-link :as Link]
            [apollo-link-context :as LinkContext]
            [apollo-link-error :as LinkError]
            [apollo-link-http :refer [HttpLink]]
            [apollo-link-state :as StateLink]
            [apollo.core :as a]
            [burningswell.web.fragments.matcher :refer [fragment-matcher]]
            [burningswell.web.location :as location]
            [burningswell.web.logging :as log]
            [burningswell.web.storage :as storage]
            [burningswell.web.ui.auto-complete :as auto-complete]
            [burningswell.web.ui.forms.reset-password :as reset-password]
            [burningswell.web.ui.forms.signin :as signin]
            [burningswell.web.ui.forms.signup :as signup]
            [burningswell.web.viewport :as viewport]
            [clojure.walk :as walk]
            [goog.object :as obj]
            [no.en.core :refer [format-url]]
            [node-fetch :as fetch]))

(def logger
  "The logger of the current namespace."
  (log/logger (namespace ::logger)))

(def cache-redirects
  #js {:Query
       #js {:spot
            (fn [_ args opts]
              ((a/get opts :getCacheKey)
               #js {:__typename "Spot"
                    :id (a/get args :id)}))}})

(defn cache [& [opts]]
  (let [cache (InMemoryCache.
               #js {:cacheRedirects cache-redirects
                    :fragmentMatcher fragment-matcher})]
    (when (exists? js/window)
      (.restore cache js/window.__APOLLO_STATE__))
    cache))

(def default-drawer-status
  {:__typename "Drawer"
   :open false})

(defn- default-route-state
  [{:keys [handler uri]}]
  {:__typename "Route"
   :handler handler
   :loaded (some? handler)
   :uri uri})

(defn update-drawer! [obj args context info]
  (println "UPDATE DRAWER MUTATION")
  (println obj args context info))

(defn defaults [{:keys [defaults]}]
  {:drawer default-drawer-status
   :location (location/default-data (:location defaults))
   :route (default-route-state (:route defaults))
   :viewport (viewport/default-data (:viewport defaults))})

(defn wrap-resolver [f]
  (fn [obj args context info]
    (clj->js (f (a/js->clj obj)
                (a/js->clj args)
                (a/js->clj context)
                (a/js->clj info)))))

(defn wrap-resolvers
  [m]
  (let [f (fn [[k v]]
            (if (fn? v)
              [k (wrap-resolver v)]
              [k v]))]
    (walk/postwalk (fn [x] (if (map? x) (into {} (map f x)) x)) m)))

(def resolvers
  {:Mutation
   {:update_drawer update-drawer!
    :update_viewport viewport/update-viewport!
    ;; :update_signup_form forms/update-signup-form!
    }
   :Query
   {:auto_complete auto-complete/resolve-data
    :reset_password reset-password/resolve-data
    :signin signin/resolve-data
    :signup signup/resolve-data}})

(defn- auth-token
  "Returns the auth token from local or session storage."
  [& [opts]]
  (or (get (storage/local-storage) :auth-token)
      (get (storage/session-storage) :auth-token)))

(defn- auth-token-header
  "Returns the auth token from local or session storage."
  [& [opts]]
  (some->> (auth-token opts) (str "Bearer ")))

(defn auth-link
  "Returns the authorization link."
  [& [opts]]
  (LinkContext/setContext
   (fn [_ context]
     (let [headers (or (obj/get context "headers") #js {})]
       (obj/set headers "authorization" (auth-token-header opts))
       #js {:headers headers}))))

(defn state-link
  "Returns the Apollo client state link."
  [cache & [opts]]
  (StateLink/withClientState
   #js {:cache cache
        :defaults (clj->js (defaults opts))
        :resolvers (clj->js (wrap-resolvers resolvers))}))

(defn error-link
  "Returns the Apollo error link."
  [& [opts]]
  (LinkError/onError
   (fn [args]
     (println "ERROR LINK"))))

(defn http-link
  "Returns the Apollo HTTP link."
  [& [{:keys [uri ssr?]}]]
  (HttpLink.
   #js {:credentials "same-origin"
        :fetch (if ssr? node-fetch js/fetch)
        :uri (format-url uri)}))

(defn links
  "Returns the Apollo links by name."
  [cache & [opts]]
  {:auth (auth-link opts)
   :error (error-link opts)
   :state (state-link cache opts)
   :http (http-link opts)})

(defn link
  "Returns the Apollo link."
  [{:keys [auth error state http]}]
  (Link/from #js [auth error state http]))

(defn- on-reset-store [links]
  (.writeDefaults (:state links))
  (log/info logger "Apollo Client store successfully reset."))

(defn browser [& [opts]]
  (let [cache (cache opts)
        links (links cache opts)
        client (ApolloClient.
                #js {:cache cache
                     :link (link links)})]
    (a/on-reset-store client #(on-reset-store links))
    client))

(defn ssr [& [opts]]
  (let [cache (cache opts)
        links (links cache (assoc opts :ssr? true))
        client (ApolloClient.
                #js {:cache cache
                     :link (link links)
                     :ssrMode true})]
    (a/on-reset-store client #(on-reset-store links))
    client))
