(ns apollo.core
  (:require-macros [apollo.core :refer [resource]])
  (:refer-clojure :exclude [get js->clj])
  (:require [goog.object :as obj]
            [grafiko.lang.core :as grafiko]
            [graphql-tag :as graphql-tag]
            [react :as React :refer [createElement]]
            [react-apollo :refer [Mutation Query]]))

(defn ast*
  "Returns the JS GraphQL AST for the `doc` in s-expr format."
  [doc]
  (clj->js (grafiko/parse-document-js doc)))

(def ast
  "Memoized version of `ast*`."
  (memoize ast*))

(defn data->clj [x]
  (cond
    (number? x)
    x
    (string? x)
    x
    (nil? x)
    x
    (or (array? x) (sequential? x))
    (mapv data->clj x)

    ;; TODO: How to detect Apollo data ?
    (seq (obj/getKeys x))
    (zipmap
     (map keyword (obj/getKeys x))
     (map data->clj (obj/getValues x)))
    :else x))

(defn js->clj [x]
  (cond-> (cljs.core/js->clj x :keywordize-keys true)
    (and x (.-data x))
    (update :data data->clj)))

(defn get [x & path]
  (when x
    (if (map? x)
      (get-in x path)
      (apply obj/getValueByKeys x (map name path)))))

(defn gql [s]
  (graphql-tag s))

(defn edges
  "Returns the edges from `coll` under the key `name`."
  [coll name]
  (get coll name :edges))

(defn nodes
  "Returns the nodes from the edges of `coll` under the key `name`."
  [coll name]
  (map #(get % :node) (edges coll name)))

(defn query [opts render-fn]
  (createElement Query (clj->js (assoc opts :children #(render-fn (js->clj %))))))

(defn- wrap-mutate-fn [mutate-fn]
  (fn [opts]
    (-> (mutate-fn (clj->js opts))
        (.then #(data->clj %))
        (.catch #(js/Promise.reject (data->clj %))))))

(defn- network-error [error]
  (when-let [error (.-networkError error)]
    {:response (.-response error)
     :result (js->clj (.-result error))
     :status (.-statusCode error)}))

(defn- error [error]
  (let [network-error (network-error error)]
    (prn network-error)
    (merge (:result network-error)
           {:graphql-errors (js->clj (.-graphQLErrors error))
            :network-error network-error
            :extra-info (js->clj (.-extraInfo error))
            :message (.-message error)})))

(defn mutation [opts render-fn]
  (createElement
   Mutation
   (clj->js (assoc opts :children #(render-fn (wrap-mutate-fn %1) (js->clj %2))))))

(defn- mutate-update
  "Returns the mutation update function."
  [update-fn]
  (fn [data-proxy response]
    (update-fn data-proxy (js->clj response))))

(defn- mutate-opts [opts]
  #js {:context (-> opts :context)
       :errorPolicy (some-> opts :error-policy name)
       :mutation (:mutation opts)
       :update (some-> opts :update mutate-update)
       :variables (some-> opts :variables clj->js)})

(defn mutate! [client opts]
  (-> (.mutate client (mutate-opts opts))
      (.then #(js->clj %))
      (.catch #(js/Promise.reject (error %)))))

(defn- cache [client]
  (or (.-cache client) (:cache client)))

(defn read-fragment [client id fragment]
  (js->clj (.readFragment (cache client) #js {:fragment fragment :id id})))

(defn read-query [client query & [opts]]
  (js->clj (.readQuery
            (cache client)
            #js {:query query
                 :variables (clj->js (:variables opts))})))

(defn write-fragment [client id fragment data]
  (->> #js {:data (clj->js data)
            :fragment fragment
            :id id}
       (.writeFragment (cache client))))

(defn write-query [client query data & [opts]]
  (->> #js {:data (clj->js data)
            :query query}
       (.writeQuery (cache client)))
  data)

(defn write-data! [client data]
  (.writeData client #js {:data (clj->js data)})
  data)

(defn clear-store! [client]
  (.clearStore client))

(defn reset-store! [client]
  (.resetStore client))

(defn on-reset-store [client handler]
  (.onResetStore client handler))

(defn on-reset-store1 [client handler]
  (let [listener (.onResetStore client handler)]))
