(ns burningswell.web.ui.layout
  (:require #?(:cljs [goog.dom :as gdom])
            [burningswell.web.ui.links :as link-to]
            [burningswell.web.router :as router]
            [burningswell.web.ui.search-input :refer [search-input]]
            [clojure.string :as str]
            [rum.core :as rum]
            [rum.mdl :as mdl]))

(defn logged-in?
  "Returns true if the user is logged in, otherwise false."
  [page]
  (some? (:current-user page)))

(defn link-content [icon text]
  (list (mdl/icon {:class "layout__link_icon"} icon)
        [:span.layout__link_text text]))

(defn link-to
  "Returns a navigation link."
  [system page route icon text]
  (if (= route (-> page :route :current :name))
    [:div.layout__link.layout__link--current.mdl-navigation__link
     (link-content icon text)]
    (let [path  (router/path-for route)]
      (link-to/history
       system path (link-content icon text)
       {:class "layout__link mdl-navigation__link"}))))

(defn countries-link
  "Return the countries link."
  [system page]
  (link-to system page :countries "terrain" "Countries"))

(defn map-link
  "Return the countries link."
  [system page]
  (link-to system page :map "map" "Show Map"))

(defn settings-link
  "Return the signin link."
  [system page]
  (when (logged-in? page)
    (link-to system page :settings "settings" "Settings")))

(defn signin-link
  "Return the signin link."
  [system page]
  (link-to system page :signin "person" "Signin"))

(defn signup-link
  "Return the signup link."
  [system page]
  (when-not (logged-in? page)
    (link-to system page :signup "person_add" "Signup")))

(defn spots-link
  "Return the spots link."
  [system page]
  (link-to system page :spots "place" "Spots"))

(defn regions-link
  "Return the regions link."
  [system page]
  (link-to system page :regions "account_balance" "Regions"))

(defn class-name
  "Return the CSS class name of `page`."
  [page]
  (some-> page :route :current :name name))

(defn title-element
  "Returns the title element of the page."
  []
  #?(:cljs (aget (gdom/getElementsByTagNameAndClass "title") 0)))

(defn set-title!
  "Set the current page title."
  [page]
  #?(:cljs (when-let [element (title-element)]
             (->> ["Burning Swell" (:title page)]
                  (remove str/blank?)
                  (str/join " - ")
                  (gdom/setTextContent element)))))

(def layout-mixin
  {:did-mount
   (fn [state]
     (set-title! (-> state :rum/args second))
     state)})

(rum/defc loading < rum/static
  "Render the loading progress bar."
  [page]
  [:div.layout__loading
   (when (:loading page)
     (mdl/progress
      {:class "layout__loading-progress"
       :mdl [:indeterminate]}))])

(rum/defc layout < rum/static layout-mixin mdl/component-handler
  "Render the layout."
  [system page contents]
  [:div.layout.mdl-layout.mdl-js-layout.mdl-layout--fixed-header.mdl-layout--fixed-drawer
   {:class (class-name page)}
   [:header.mdl-layout__header

    [:div.mdl-layout__header-row

     [:span.layout__left.mdl-layout-title
      (or (:title page) "Burning Swell")]

     [:div.layout__middle
      (search-input system (assoc (:auto-complete page) :id "search-input"))]

     [:div.layout__right
      ""]]]

   [:div.layout__drawer.mdl-layout__drawer
    [:span.mdl-layout-title "Burning Swell"]
    [:hr]
    [:nav.mdl-navigation
     (countries-link system page)
     (regions-link system page)
     (spots-link system page)
     (map-link system page)
     ;; [:hr]
     ;; (settings-link system page)
     ;; (signin-link system page)
     ;; (signup-link system page)
     ]]

   [:main.mdl-layout__content
    ;; (loading page)
    contents]])
