(ns burningswell.web.ui.auto-complete
  (:require #?(:cljs [goog.events.KeyCodes :as KeyCodes])
            [rum.core :as rum]
            [rum.mdl :as mdl]))

(defn- query
  "Return the query from `state`."
  [state]
  (-> state :rum/args first))

(defn- results
  "Return the query results from `state`."
  [state]
  (-> state :rum/args second))

(defn- options
  "Return the options from `state`."
  [state]
  (-> state :rum/args (nth 2 nil)))

(defn- index
  "Return the index from `state`."
  [state]
  (-> state :auto-complete deref :index))

(defn- set-index!
  "Set the auto complete index."
  [state index]
  (swap! (:auto-complete state) assoc :index index))

(defn- set-show-results!
  "Set the :show-results? flag."
  [state show-results?]
  (swap! (:auto-complete state) assoc :show-results? show-results?))

(defn- set-loading!
  "Set the :loading? flag."
  [state loading?]
  (swap! (:auto-complete state) assoc :loading? loading?))

(defn- on-clear-query
  "Clear the current query."
  [state opts]
  (set-index! state -1)
  (set-show-results! state false)
  (set-loading! state false)
  (when-let [handler (:on-change opts)]
    (handler nil)))

(defn- select-result
  "Handle selecting a search result."
  [state result]
  (when-let [handler (-> state options :on-select)]
    (set-show-results! state false)
    (handler result)))

(defn- on-select
  "Handle selecting a search result."
  [state]
  (select-result state (nth (results state) (index state))))

(defn- on-submit
  "Handle submitting a query."
  [state]
  (when-let [handler (-> state options :on-submit)]
    (handler (query state))))

(defn- on-enter-key
  "Handle enter key."
  [state]
  (set-show-results! state false)
  (set-loading! state true)
  (if (neg? (index state))
    (on-submit state)
    (on-select state)))

(defn- on-escape-key
  "Handle escape key."
  [state]
  (set-show-results! state false)
  (set-loading! state false))

(defn- select-next
  "Move to the next auto complete result."
  [state]
  (let [next (inc (index state))]
    (set-show-results! state true)
    (if (< next (count (results state)))
      (set-index! state next)
      (set-index! state -1))))

(defn- select-previous
  "Move to the previous auto complete result."
  [state]
  (let [prev (dec (index state))
        results (results state)]
    (set-show-results! state true)
    (if (= prev -2)
      (set-index! state (dec (count results)))
      (set-index! state prev))))

(defn- on-query-change
  "Handle query changes."
  [state opts event]
  (set-index! state -1)
  (set-show-results! state true)
  (set-loading! state true)
  (when-let [handler (:on-change opts)]
    (handler (.-value (.-target event)))))

(defn- on-key-down
  "Handle key down events."
  [state event]
  #?(:cljs (condp = (.-keyCode event)
             KeyCodes/DOWN (select-next state)
             KeyCodes/ENTER (on-enter-key state)
             KeyCodes/ESC (on-escape-key state)
             KeyCodes/UP (select-previous state)
             nil)))

(def auto-complete-mixin
  "The auto complete mixin."
  {:did-remount
   (fn [old-state new-state]
     (when (or (empty? (results new-state))
               (not= (results old-state) (results new-state))
               (not= (query old-state) (query new-state)))
       #?(:cljs (js/setTimeout #(set-loading! new-state false) 500)))
     new-state)})

(def auto-complete-state
  "The auto complete local state."
  (rum/local {:index -1 :show-results? false :loading? false} :auto-complete))

(rum/defcs auto-complete < rum/static auto-complete-mixin auto-complete-state
  "Render an auto complete input field."
  [state query results opts]
  (let [{:keys [index loading? show-results?]} @(:auto-complete state)
        clear-id (when (:id opts) (str (:id opts) "__clear"))]
    [:div.auto-complete
     [:div.auto-complete__icon-search
      (if loading?
        (mdl/spinner {:mdl [:single-color] :is-active true})
        (mdl/icon "search"))]
     [:input.auto-complete__query
      {:autoFocus (:auto-focus opts)
       :type "text"
       :on-change #(on-query-change state opts %)
       :on-click #(set-show-results! state true)
       :on-key-down #(on-key-down state %)
       :placeholder (:placeholder opts)
       :value (or query "")
       :ref "query"}]
     (mdl/button
      {:class #{"auto-complete__clear" "mdl-button--icon"}
       :id clear-id
       :on-click #(on-clear-query state opts)}
      (mdl/icon "clear"))
     (mdl/tooltip {:mdl [:bottom] :for clear-id} "Clear search")
     [:div.auto-complete__suggestions
      {:class (if (or (empty? results)
                      (not show-results?))
                "auto-complete__suggestions--hidden")}
      (for [[current result] (map-indexed vector results)]
        [:div.auto-complete__result
         {:class (if (= current index) "auto-complete__result--highlight")
          :key (str "auto-complete__result--" current)
          :on-click #(select-result state result)
          :on-mouse-enter #(set-index! state current)
          :on-touch-end #(select-result state result)}
         ((or (:render opts) pr-str) result)])]]))
