(ns burningswell.web.components.autocomplete
  (:require [burningswell.web.geocoder :as geocoder]
            [burningswell.web.mixins.event-handler :as events]
            [burningswell.web.util :as util]
            [clojure.string :as str]
            [goog.events.KeyCodes :as KeyCodes]
            [goog.style :as style]
            [om-tools.core :refer-macros [defcomponentk]]
            [om.core :as om]
            [sablono.core :as html :refer-macros [html]])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn- set-index!
  "Set the auto complete index."
  [owner index]
  (om/set-state! owner :index index))

(defn- on-change
  "Handle auto complete change events."
  [owner event]
  (let [completion-fn (om/get-state owner :completion-fn)
        value (.-value (.-target event))]
    (om/set-state! owner :value value)
    (om/set-state! owner :show-results true)
    (when (and completion-fn (not (str/blank? value)))
      (set-index! owner -1)
      (go (let [results (<! (completion-fn value))]
            (om/set-state! owner :results results))))))

(defn- on-select
  "Handle auto complete selection."
  [owner index]
  (let [{:keys [results select-fn value value-fn]}
        (om/get-state owner)]
    (om/set-state! owner :show-results false)
    (if-let [result (nth results index nil)]
      (let [value (value-fn result)]
        (select-fn
         {:value value
          :selected result})
        (om/set-state! owner :value value))
      (select-fn {:value value}))))

(defn- on-enter
  "Handle auto complete enter events."
  [owner event]
  (let [index (om/get-state owner :index)]
    (on-select owner index)
    (.preventDefault event)))

(defn- on-touch-start
  "Handle auto complete touch start events."
  [owner index]
  (on-select owner index))

(defn- next-result
  "Move to the next auto complete result."
  [owner]
  (let [next (inc (om/get-state owner :index))
        results (om/get-state owner :results)]
    (om/set-state! owner :show-results true)
    (if (< next (count results))
      (set-index! owner next)
      (set-index! owner -1))))

(defn- previous-result
  "Move to the previous auto complete result."
  [owner]
  (let [prev (dec (om/get-state owner :index))
        results (om/get-state owner :results)]
    (om/set-state! owner :show-results true)
    (if (= prev -2)
      (set-index! owner (dec (count results)))
      (set-index! owner prev))))

(defn- on-key-down
  "Handle auto complete key down events."
  [owner event]
  (condp = (.-keyCode event)
    KeyCodes/ENTER
    (on-enter owner event)
    KeyCodes/ESC
    (om/set-state! owner :show-results false)
    KeyCodes/UP
    (previous-result owner)
    KeyCodes/DOWN
    (next-result owner)
    nil))

(defn- on-blur
  "Handle auto complete blur events."
  [owner event]
  (om/set-state! owner :show-results false))

(defn- update-size
  "Update the size of the auto complete drop down."
  [owner]
  (let [size (style/getSize (om/get-node owner "input"))
        size {:width (.-width size) :height (.-height size)}]
    (om/update-state! owner #(merge % size))))

(defn render-results [owner]
  (let [{:keys [index render-fn results show-results width]}
        (om/get-state owner)]
    (when show-results
      [:div.autocomplete__results
       {:style
        {:position "absolute"
         :z-index 1000}}
       (for [[current result] (map-indexed vector results)]
         [:div.autocomplete__result
          {:class (if (= current index)
                    "autocomplete__result--current")
           :on-touch-start #(on-touch-start owner current)
           :style {:width width}}
          (render-fn result)])])))

(defcomponentk autocomplete
  [data owner state opts]
  (:mixins events/handler)
  (init-state [_]
    {:completion-fn (:completion-fn opts)
     :height nil
     :index -1
     :name (:name opts)
     :render-fn (or (:render-fn opts) pr-str)
     :select-fn (or (:select-fn opts) prn)
     :value-fn (or (:value-fn opts) str)
     :show-results false
     :value data
     :width nil})
  (did-mount [_]
    (update-size owner)
    (events/listen owner js/window :resize #(update-size owner)))
  (did-update [_ _ _]
    (update-size owner))
  (render [_]
    (html
     [:div.autocomplete
      (if-let [label (:label opts)]
        [:label.autocomplete__label label])
      (html/text-field
       {:auto-complete "off"
        :class "autocomplete__input"
        :on-blur #(on-blur owner %)
        :on-change #(on-change owner %)
        :on-key-down #(on-key-down owner %)
        :placeholder (:placeholder opts)
        :ref "input"}
       (:name @state)
       (:value @state))
      (render-results owner)])))

;; Address auto completion

(defn render-address [owner address]
  (html
   [:div.autocomplete__address
    (util/country-flag (:country address))
    (:formatted address)]))

(defcomponentk address
  "Return an auto completing address component."
  [data owner opts state]
  (render [_]
    (->autocomplete
     data {:opts
           (merge
            {:completion-fn geocoder/geocode-address
             :placeholder "Type in an address"
             :render-fn #(render-address owner %)
             :value-fn :formatted}
            opts)})))
