(ns burningswell.web.server
  (:require [burningswell.pedestal.server :as pedestal]
            [burningswell.transit :as transit]
            [burningswell.web.cookies :as cookies]
            [burningswell.web.interceptors :as i]
            [burningswell.web.modules.core :as modules]
            [burningswell.web.modules.countries]
            [burningswell.web.modules.country]
            [burningswell.web.modules.map]
            [burningswell.web.modules.new_spot]
            [burningswell.web.modules.region]
            [burningswell.web.modules.regions]
            [burningswell.web.modules.root]
            [burningswell.web.modules.search]
            [burningswell.web.modules.settings]
            [burningswell.web.modules.signin]
            [burningswell.web.modules.signup]
            [burningswell.web.modules.signout]
            [burningswell.web.modules.spot]
            [burningswell.web.modules.spots]
            [burningswell.web.modules.welcome]
            [burningswell.web.system.client :as client]
            [burningswell.web.system.core :as system]
            [clojure.core.async :as async]
            [clojure.core.async.impl.protocols :as proto]
            [clojure.string :as str]
            [com.stuartsierra.component :as component]
            [coolant.core :as coolant]
            [geo.postgis :as geo]
            [hiccup.core :refer [html]]
            [hiccup.def :refer [defhtml]]
            [hiccup.element :refer [javascript-tag]]
            [hiccup.page :refer [html5 include-css include-js]]
            [io.pedestal.http :as http]
            [io.pedestal.http.body-params :as body-params]
            [io.pedestal.http.route :as route]
            [io.pedestal.http.route.definition :refer [defroutes]]
            [io.pedestal.interceptor :refer [interceptor]]
            [ring.util.response :refer [response]]
            [rum.core :as rum]))

(defn sanitize-script
  "Sanitize all script tags within `s`."
  [s]
  (when s (str/replace s "</" "<\\/")))

(defhtml include-js-async
  "Include a list of external javascript files."
  [script async?]
  [:script
   {:async (true? async?)
    :type "text/javascript"
    :src script}])

(defn client-config
  "Return the client config."
  [config]
  (-> (select-keys config [:api-client :elements])
      (assoc :facebook (select-keys (:facebook config) [:client-id :redirect-uri]))))

(defhtml include-config
  "Include `config` in a script tag."
  [config]
  [:script
   {:id (-> config :elements :config)
    :type "application/transit+json"}
   (-> (client-config config)
       (transit/encode)
       (sanitize-script))])

(defn include-app
  "Include the JavaScript for the web application."
  [config]
  (include-js-async "/javascripts/burningswell.js" (:async-js config)))

(defn include-google-maps
  "Include the JavaScript for Google Maps."
  [config]
  (include-js-async
   (format "https://maps.googleapis.com/maps/api/js?key=%s"
           (-> config :google :maps :api-key))
   true))

(defhtml include-state
  "Include the system state in a script tag."
  [system]
  [:script
   {:id (-> system :config :elements :state)
    :type "application/transit+json"}
   (-> (transit/encode (-> system :coolant deref :state))
       (sanitize-script))])

(defn start-app
  "Start the web application."
  [config]
  (-> (str "
       window.addEventListener(\"load\", function() {
         burningswell.web.main.start(\"%s\");
       });")
      (format (-> config :elements :config))
      (javascript-tag)))

(defhtml header
  "Render the page header."
  []
  [:head {:lang "en"}
   [:title "Burning Swell"]
   [:meta {:charset "UTF-8"}]
   [:meta {:http-equiv "Accept-CH" :content "DPR, Width, Viewport-Width"}]
   [:meta {:http-equiv "X-UA-Compatible" :content "IE=edge"}]
   [:meta {:name "mobile-web-app-capable" :content "yes"}]
   [:meta {:name "viewport" :content "width=device-width, initial-scale=1.0"}]
   ;; Application Manifest
   [:link {:rel "manifest" :href "/manifest.json"}]
   [:link {:rel "icon" :href "/images/launcher/icon-4x.png" :sizes "192x192"}]
   ;; Application Manifest for Safari on iOS
   [:meta {:name "apple-mobile-web-app-capable" :content "yes"}]
   [:meta {:name "apple-mobile-web-app-status-bar-style" :content "black"}]
   [:meta {:name "apple-mobile-web-app-title" :content "Burning Swell"}]
   [:link {:rel "apple-touch-icon" :href "/images/launcher/icon-4x.png"}]
   ;; Application Manifest for Windows
   [:meta {:name "msapplication-TileImage", :content "/images/launcher/icon-4x.png"}]
   [:meta {:name "msapplication-TileColor", :content "#2F3BA2"}]
   ;; Stylesheets
   (include-css "https://fonts.googleapis.com/css?family=Roboto")
   (include-css "https://fonts.googleapis.com/icon?family=Material+Icons")
   (include-css "/stylesheets/burningswell.min.css")])

(defn page
  "Render the page."
  [{:keys [config] :as system} & [content]]
  (response
   (html5
    (header)
    [:body
     [:div {:id (-> config :elements :root)} content]
     (include-google-maps config)
     (include-js-async "/javascripts/diaporama.min.js" true)
     (include-config config)
     (include-state system)
     (include-app config)
     (start-app config)])))

(defn geocoder-location
  "Return the geocoder location from `request`."
  [request]
  (let [latitude (-> request :location :latitude)
        longitude (-> request :location :longitude)]
    (when (and latitude longitude)
      (geo/point 4326 longitude latitude))))

(defn setup-location
  "Read the location from the `request` cookie and dispatch an event."
  [system request]
  (some->> (or (geocoder-location request)
               (cookies/location request))
           (system/change-current-location system)))

(defn setup-route [system request]
  (let [channel (system/change-route system (:route request))]
    (when (satisfies? proto/Channel channel)
      (async/<! channel))))

(defn setup-viewport-size
  "Read the viewport size from the `request` cookie and dispatch an event."
  [system request]
  (some->> (cookies/viewport-size request)
           (system/viewport-size-changed system)))

(def index
  (interceptor
   {:name ::index
    :enter
    (fn [{:keys [request] :as context}]
      (let [system (component/start (client/client (:config request)))]
        (async/go
          (try
            (setup-location system request)
            (setup-viewport-size system request)
            (let [channel (system/change-route system (:route request))]
              (when (satisfies? proto/Channel channel)
                (async/<! channel))
              (async/<! (system/signin-from-route system (:route request)))
              (let [response (page system (modules/render-server system))]
                (component/stop system)
                (assoc context :response response)))
            (catch Exception e
              (.printStackTrace e)
              (page system))))))}))

(defn devcards
  "Render the devcards page."
  [{:keys [config] :as request}]
  (response
   (html5
    (header)
    [:body
     (include-js (format "https://maps.googleapis.com/maps/api/js?key=%s" (-> config :google :maps :api-key)))
     (include-js "https://storage.googleapis.com/code.getmdl.io/1.1.3/material.min.js")
     (include-js "/javascripts/diaporama.min.js")
     (include-config config)
     (include-js "/javascripts/burningswell.devcards.js")])))

(defroutes routes
  [[["/" ^:interceptors [http/html-body i/cookies i/route i/geoip]
     {:get [:welcome index]}
     ["/devcards" {:get [:devcards devcards]}]
     ["/countries" {:get [:countries index]}]
     ["/countries/:id"
      ^:constraints {:id #"[0-9]+"}
      {:get [:country index]}]
     ["/map" {:get [:map index]}]
     ["/regions" {:get [:regions index]}]
     ["/regions/:id"
      ^:constraints {:id #"[0-9]+"}
      {:get [:region index]}]
     ["/search" {:get [:search index]}]
     ["/settings" {:get [:settings index]}]
     ["/signin" {:get [:signin index]}]
     ["/signout" {:get [:signout index]}]
     ["/signup" {:get [:signup index]}]
     ["/spots" {:get [:spots index]}]
     ["/spots/:id"
      ^:constraints {:id #"[0-9]+"}
      {:get [:spot index]}]]]])

(defn new-server
  "Return a new server."
  [config]
  (-> (assoc config :init i/init :routes routes)
      (pedestal/server)
      (component/using [:config :geoip])))
