(ns burningswell.web.modules.spots
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go]]))
  (:require [burningswell.web.api :as api]
            [burningswell.web.coolant :as coolant]
            [burningswell.web.getter.spots :as spots]
            [burningswell.web.modules.core :refer [module-loaded render-server]]
            [burningswell.web.stores.spots :as store]
            [burningswell.web.system.core :as system]
            [burningswell.web.ui.layout :as layout]
            [burningswell.web.ui.spot-list :as spot-list]
            [clojure.core.async :refer [<! #?(:clj go)]]
            [rum.core :as rum]))

(defn spots-params
  "Return the query params for the spots page."
  [system & [params]]
  (merge (system/current-location-params system)
         (api/pagination system :collection)
         params))

(defn load-spots
  "Fetch the spots."
  [system & [params]]
  (let [params (spots-params system params)]
    (system/dispatch! system :spots/load-spots-start params)
    (go (let [{:keys [status body]} (<! (api/spots system params))]
          (case status
            200 (system/dispatch! system :spots/load-spots-success body)
            (system/dispatch! system :spots/load-spots-fail body))))))

(defn load-next [system]
  (let [page (coolant/get system spots/page)]
    (when-not (:loading? page)
      (load-spots system {:page (inc (:page page))}))))

(rum/defc content < rum/static
  "Render the content of the spots page."
  [system {:keys [spots] :as page}]
  (layout/layout
   system page
   [:div.spots__content
    (spot-list/infinite-spot-list
     system spots
     {:on-load #(load-next system)
      :scroll-target layout/scroll-target})]))

(rum/defcs page < (coolant/mixin spots/page)
  "Render the spots page."
  [page system]
  (content system page))

(defmethod system/change-route :spots [system route]
  (coolant/register-stores! system [store/store])
  (system/route-changed system route)
  (load-spots system (:query-params route)))

(defmethod render-server :spots [system]
  #?(:clj (->> (coolant/get system spots/page)
               (content system)
               (rum/render-html))))

(def ^:export main page)
(module-loaded :spots)
