(ns burningswell.web.modules.signup
  "The signup page."
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go]]))
  (:require [burningswell.web.api :as api]
            [burningswell.web.coolant :as coolant]
            [burningswell.web.dom :as dom]
            [burningswell.web.getter.signup :as signup]
            [burningswell.web.modules.core :refer [module-loaded render-server]]
            [burningswell.web.stores.signup :as store]
            [burningswell.web.system.core :as system]
            [burningswell.web.system.history :as history]
            [burningswell.web.ui.facebook :as facebook]
            [burningswell.web.ui.image :refer [image]]
            [burningswell.web.ui.layout :refer [layout]]
            [burningswell.web.ui.links :as link-to]
            [clojure.core.async :refer [<! timeout #?(:clj go)]]
            [clojure.string :as str]
            [rum.core :as rum]
            [rum.mdl :as mdl]))

(defn on-change [system field]
  (fn [event]
    (system/dispatch! system field (dom/value (.-target event)))
    (go (let [form (coolant/evaluate system signup/form)
              {:keys [status body]} (<! (api/validate-user system form))]
          (system/dispatch! system :signup/errors body)))))

(rum/defc email < rum/static
  "Render the email form field."
  [system email errors submitted?]
  (mdl/textfield
   {:class "signup__email"
    :mdl [:floating-label]}
   (rum/with-key
     (mdl/textfield-input
      {:auto-complete "off"
       :id "signup__email"
       :name "email"
       :on-change (on-change system :signup/email)
       :value (str email)})
     "signup-email")
   (mdl/textfield-label
    {:for "signup__email"}
    "Your email address")
   (mdl/textfield-error
    (when (or email submitted?)
      (first errors)))))

(rum/defc username < rum/static
  "Render the username form field."
  [system username errors submitted?]
  (mdl/textfield
   {:class "signup__username"
    :mdl [:floating-label]}
   (rum/with-key
     (mdl/textfield-input
      {:auto-complete "off"
       :id "signup__username"
       :name "username"
       :on-change (on-change system :signup/username)
       :value (str username)})
     "signup-username")
   (mdl/textfield-label
    {:for "signup__username"}
    "Choose your username")
   (mdl/textfield-error
    (when (or username submitted?)
      (first errors)))))

(rum/defc password < rum/static
  "Render the password form field."
  [system password errors submitted?]
  (mdl/textfield
   {:class "signup__password"
    :mdl [:floating-label]}
   (rum/with-key
     (mdl/textfield-input
      {:auto-complete "off"
       :id "signup__password"
       :name "password"
       :on-change (on-change system :signup/password)
       :type "password"
       :value (str password)})
     "signup-password")
   (mdl/textfield-label
    {:for "signup__password"}
    "Choose a secure password")
   (mdl/textfield-error
    (when (or password submitted?)
      (first errors)))))

(defn signin
  "Sign a user in."
  [system user]
  (go (let [{:keys [status body]} (<! (api/create-jws-token system user))]
        (system/dispatch! system :signin/submit true)
        (case status
          201 (do (system/dispatch! system :signin/clear)
                  (when (<! (system/signin-with-token system (:token body)))
                    (history/set-token! (:history system) "/")))
          401 (do (system/dispatch! system :signin/errors ["Incorrect credentials. Try again!"])
                  (system/dispatch! system :signin/shake true)
                  (<! (timeout 1000))
                  (system/dispatch! system :signin/shake false))))))

(defn submit-form
  "Submit the signup form."
  [system user event]
  (.preventDefault event)
  (go (let [{:keys [status body]} (<! (api/create-user system user))]
        (system/dispatch! system :signup/submit true)
        (case status
          201 (do (system/dispatch! system :signup/clear body)
                  (signin system {:login (:username user)
                                  :password (:password user)}))
          422 (do (system/dispatch! system :signup/errors (:errors body))
                  (system/dispatch! system :signup/shake true)
                  (<! (timeout 1000))
                  (system/dispatch! system :signup/shake false))))))

(rum/defc submit-button < rum/static
  "Render the signup button."
  [system form]
  (mdl/button
   {:class "signup__submit"
    :mdl [:colored :raised :ripple]
    :on-click #(submit-form system form %)}
   "Create Account"))

(rum/defc help < rum/static
  "Render the signup help."
  [system]
  [:div.signup__member
   "Already a member?" (link-to/login system)])

(defn facebook-button
  "Render the Facebook signup button."
  [system]
  (facebook/login-button
   (-> system :config :facebook)
   "Sign up with Facebook"
   {:class "signup__facebook"}))

(rum/defc signup-form < rum/static
  "Render the signup form."
  [system form errors]
  [:form.signup__form
   {:on-submit #(submit-form system form %)}
   (email system (:email form) (:email errors) (:submitted? form))
   (username system (:username form) (:username errors) (:submitted? form))
   (password system (:password form) (:password errors) (:submitted? form))
   (submit-button system form)
   [:div.signup__or [:span "or"]]
   (facebook-button system)
   (help system)])

(rum/defc signup-card < rum/static
  "Render the signup card."
  [system page]
  (mdl/card
   {:class ["signup__card" (if (:shake? page) "signup__card--shake")]}
   (mdl/card-title "Sign Up")
   (mdl/card-text "Create a new account.")
   (signup-form system (:form page) (:errors page))))

(defn signup-background
  "Render the background image of the signup page."
  []
  (image
   {:class "signup__image"
    :background "black"
    :preload true
    :sizing "cover"
    :fade true
    :src "https://farm9.staticflickr.com/8113/8683411065_e546bcdd3b_o.jpg"}))

(rum/defc content < rum/static
  "Render the content of the signup page."
  [system page]
  (layout
   system page
   [:div.signup__content
    (signup-background)
    (signup-card system page)]))

(rum/defcs page < (coolant/mixin signup/page)
  "Render the signup page."
  [page system]
  (content system page))

(defmethod system/change-route :signup [system route]
  (coolant/register-stores! system [store/store])
  (system/route-changed system route))

(defmethod render-server :signup [system]
  #?(:clj (->> (coolant/get system signup/page)
               (content system)
               (rum/render-html))))

(def ^:export main page)
(module-loaded :signup)
