(ns burningswell.web.modules.map
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go]]))
  (:require [burningswell.web.api :as api]
            [burningswell.web.coolant :as coolant]
            [burningswell.web.getter.map :as map]
            [burningswell.web.modules.core :refer [module-loaded render-server]]
            [burningswell.web.stores.map :as store]
            [burningswell.web.system.core :as system]
            [burningswell.web.system.history :as history]
            [burningswell.web.system.router :as router]
            [burningswell.web.ui.google-map :as google-map]
            [burningswell.web.ui.layout :refer [layout]]
            [burningswell.web.ui.search-list :refer [search-list]]
            [burningswell.web.util :as util]
            [clojure.core.async :refer [<! #?(:clj go)]]
            [rum.core :as rum]))

(defn save-map-center
  "Save the map center to local storage."
  [system center]
  (assoc! (:local-storage system) :map-center center))

(defn save-map-zoom
  "Save the map zoom level to local storage."
  [system zoom]
  (assoc! (:local-storage system) :map-zoom zoom))

(defn change-map-bounding-box
  "Change the bounding box of the map."
  [system bounding-box]
  (system/dispatch! system :map/bounding-box bounding-box))

(defn change-map-center
  "Change the center of the map."
  [system center]
  (system/dispatch! system :map/center center)
  (save-map-center system center))

(defn change-map-zoom
  "Change the zoom level of the map."
  [system zoom]
  (system/dispatch! system :map/zoom zoom)
  (save-map-zoom system zoom))

(defn update-map-query-params
  "Update the map query params."
  [system center zoom]
  (->> (assoc (util/location-query-params center) :zoom zoom)
       (router/path-for :map)
       (history/replace-state! (:history system))))

(defn map-idle
  "Fetch countries, regions or spots bases on zoom level."
  [system bounding-box center zoom]
  (system/dispatch! system :page/loading true)
  (update-map-query-params system center zoom)
  (let [params (merge (util/bounding-box-query-params bounding-box) {:zoom zoom :min-spots 1})]
    (go (let [params (merge (api/pagination system :map) params)
              {:keys [status body]} (<! (api/map system params))]
          (case status
            200 (system/dispatch! system :map/results body))
          (system/dispatch! system :page/loading false)))))

(defn map-location
  "Return the current map location."
  [system page]
  (or (-> page :map :center)
      (-> system :local-storage :map-center)
      (-> system :local-storage :location)))

(defn map-zoom
  "Return the current map zoom level."
  [system page]
  (or (-> page :map :zoom)
      (-> system :local-storage :map-zoom)))

(defn on-map-idle
  "Handle map idle events."
  [system page map]
  (change-map-center system (google-map/center map))
  (change-map-bounding-box system (google-map/bounding-box map))
  (map-idle
   system
   (google-map/bounding-box map)
   (google-map/center map)
   (google-map/zoom map)))

(defn on-center-changed
  "Handle center change events."
  [system page map]
  (change-map-center system (google-map/center map)))

(defn on-zoom-changed
  "Handle zoom change events."
  [system page map]
  (change-map-zoom system (google-map/zoom map)))

(rum/defc content
  "Render the content of the map page."
  [system {:keys [results] :as page}]
  (layout
   system page
   [:div.map__content
    (google-map/google-map
     {:class "map__map"
      :location (map-location system page)
      :zoom (map-zoom system page)
      :on-idle #(on-map-idle system page %)
      :on-zoom-changed #(on-zoom-changed system page %)})
    (search-list system results)]))

(rum/defcs page < (coolant/mixin map/page)
  "Render the map page."
  [page system]
  (content system page))

(defmethod system/change-route :map [system route]
  (coolant/register-stores! system [store/store])
  (system/route-changed system route))

(defmethod render-server :map [system]
  #?(:clj (->> (coolant/get system map/page)
               (content system)
               (rum/render-html))))

(def ^:export main page)
(module-loaded :map)
