(ns burningswell.web.ui.infinite-list
  (:require #?(:cljs [goog.dom :as gdom])
            #?(:cljs [goog.style :as style])
            [burningswell.web.logging :as log]
            [burningswell.web.ui.mixins.event-handler :as events]
            [rum.core :as rum]))

(def logger
  "The logger of the current namespace."
  (log/logger "burningswell.web.ui.infinite-list"))

(def default-opts
  "The default options."
  {:loading false
   :per-page 10
   :threshold 100})

(defn page-height
  "Return the height of the page."
  []
  #?(:cljs js/document.documentElement.clientHeight))

(defn scroller
  "Return the scroller of `event`."
  [state event]
  (let [target (.-target event)]
    (or (aget target "scroller") target)))

(defn scroll-top
  "Return the top position of the `event` scroller."
  [state event]
  (.-scrollTop (scroller state event)))

(defn scroll-content-height
  "Return the height of the `event` scroller."
  [state event]
  (.-scrollHeight (scroller state event)))

(defn scroll-header-height
  "Return the header height of the `event` scroller."
  [state event]
  #?(:cljs (when-let [header (.-header (.-target event))]
             (.-height (style/getSize header)) 0)))

(defn scroll-visible-height
  "Return the content height of the `event` scroller."
  [state event]
  (- (page-height) (scroll-header-height state event)))

(defn content-height
  "Return the height of the content."
  [state]
  #?(:cljs (let [node (rum/dom-node state)]
             (.-height (style/getSize node)))))

(defn bottom-reached?
  "Returns true if the scroll position is near the bottom of the page."
  [state event]
  (neg? (- (scroll-content-height state event)
           (scroll-visible-height state event)
           (scroll-top state event)
           (:threshold state))))

(defn load-next
  "Fetch the next page."
  [state]
  (when-let [handler (:on-load state)]
    (handler)))

(defn on-scroll
  "The scroll event handler."
  [state event]
  (when (bottom-reached? state event)
    (log/debug logger "Infinite list bottom reached.")
    (load-next state)))

(defn on-resize
  "The resize event handler."
  [state event]
  (when (< (content-height state) (page-height))
    (load-next state)))

(defn scroll-target
  "Return the scroll target from `state`."
  [{:keys [scroll-target] :as state}]
  #?(:cljs (cond
             (nil? scroll-target)
             js/window
             (string? scroll-target)
             (or (some-> (js/document.getElementsByTagName scroll-target)
                         array-seq first)
                 (gdom/getElementByClass scroll-target))
             (ifn? scroll-target)
             (scroll-target)
             :else scroll-target)))

(defn attach-listeners
  "Attach scroll and resize listeners."
  [{:keys [scroller] :as state}]
  #?(:cljs (when-let [target (scroll-target state)]
             (events/listen state js/window :resize #(on-resize state %))
             (events/listen state target :resize #(on-resize state %))
             (events/listen state target :scroll #(on-scroll state %))
             state)))

(def infinite-list-mixin
  "The infinite list mixin."
  {:init
   (fn [state props]
     (let [[list-items opts] (:rum/args state)]
       (merge state default-opts opts)))
   :did-mount
   (fn [state]
     (attach-listeners state))})

(rum/defcs infinite-list < rum/static < events/mixin < infinite-list-mixin
  "Render an infinite list."
  [state list-items & [opts]]
  [:div.infinite-list
   {:class (:class opts)}
   [:div.infinite-list__header
    {:class (:header-class opts)}
    (:header opts)]
   [:div.infinite-list__content
    {:class (:list-class opts)}
    (for [[index list-item] (map-indexed vector list-items)]
      [:div.infinite-list__item
       {:class (:item-class opts)
        :key (str "infinite-list__key-" index)}
       list-item])]
   [:div.infinite-list__footer
    {:class (:footer-class opts)}
    (:footer opts)]])
