(ns burningswell.web.modules.countries
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go]]))
  (:require [burningswell.web.api :as api]
            [burningswell.web.coolant :as coolant]
            [burningswell.web.getter.countries :as countries]
            [burningswell.web.modules.core :refer [module-loaded render-server]]
            [burningswell.web.system.core :as system]
            [burningswell.web.ui.country-list :refer [infinite-country-list]]
            [burningswell.web.ui.layout :as layout]
            [clojure.core.async :refer [<! #?(:clj go)]]
            [rum.core :as rum]))

(defn countries-params
  "Return the query params for the countries page."
  [system & [params]]
  (merge {:min-spots 1}
         (system/current-location-params system)
         (api/pagination system :collection params)
         params))

(defn load-countries
  "Fetch the countries."
  [system & [params]]
  (let [params (countries-params system params)]
    (system/dispatch! system :countries/load-countries-start params)
    (go (let [{:keys [status body]} (<! (api/countries system params))]
          (case status
            200 (system/dispatch! system :countries/load-countries-success body)
            (system/dispatch! system :countries/load-countries-fail body))))))

(defn load-next [system]
  (let [page (coolant/get system countries/page)]
    (when-not (:loading? page)
      (load-countries system {:page (inc (:page page))}))))

(rum/defc content < rum/static
  "Render the content of the countries page."
  [system {:keys [countries] :as page}]
  (layout/layout
   system page
   [:div.countries__content
    (infinite-country-list
     system countries
     {:on-load #(load-next system)
      :scroll-target layout/scroll-target})]))

(rum/defcs page < (coolant/mixin countries/page)
  "Render the countries page."
  [page system]
  (content system page))

(defmethod system/change-route :countries [system route]
  (system/route-changed system route)
  (load-countries system (:query-params route)))

(defmethod render-server :countries [system]
  #?(:clj (->> (coolant/get system countries/page)
               (content system)
               (rum/render-html))))

(def ^:export main page)
(module-loaded :countries)
