(ns burningswell.web.api
  (:refer-clojure :exclude [map])
  (:require [burningswell.api.client :as api]
            [burningswell.web.getter.core :as getter]
            [coolant.core :as coolant]
            [request.core :as http]))

(def pagination-params
  "The pagination params based on breakpoint type."
  {:collection {:xxl 42 :xl 36 :l 30 :m 25 :s 18 :xs 12 :xxs 8}
   :map {:xxl 28 :xl 24 :l 20 :m 16 :s 15 :xs 10 :xxs 8}
   :spots-around {:xxl 15 :xl 12 :l 9 :m 6 :s 6 :xs 12 :xxs 1}})

(defn pagination
  "Return the pagination params for the spots around a spot."
  [system type & [params]]
  (let [layout (:layout (coolant/evaluate (:coolant system) getter/layout))
        per-page (or (get-in pagination-params [type layout]) 16)]
    (merge {:page 1 :per-page per-page} params)))

(defn api
  "Return the API client from `system`."
  [system]
  (if-let [token (coolant/evaluate (:coolant system) getter/auth-token)]
    (assoc (:api-client system) :auth-token token)
    (:api-client system)))

(defn create-user
  "Create a new user."
  [system user]
  (http/post (:api-client system) :users {:transit-params user}))

(defn create-jws-token
  "Create a new JWS token."
  [system user]
  (let [auth [(:login user) (:password user)]]
    (http/post (:api-client system) :jwt-token {:basic-auth auth})))

(defn create-session
  "Create a new surf session."
  [system session]
  (http/post (api system) :sessions {:transit-params session}))

(defn create-spot
  "Create a new spot."
  [system user spot]
  (->> {:auth-token (:auth-token user)
        :transit-params spot}
       (http/post (api system) :spots)))

(defn countries
  "Fetch countries."
  [system & [params]]
  (http/get (api system) :countries {:query-params params}))

(defn country
  "Fetch the `country`."
  [system country & [params]]
  (http/get (api system) :country country {:query-params params}))

(defn map
  "Fetch countries, regions or spots based on the zoom query parameter."
  [system & [params]]
  (http/get (api system) :map {:query-params params}))

(defn me
  "Fetch the currently logged in user."
  [system token & [params]]
  (http/get (:api-client system) :me {:headers {"authorization" (str "Token " token)}}))

(defn region
  "Fetch the `region`."
  [system region & [params]]
  (http/get (api system) :region region {:query-params params}))

(defn regions
  "Fetch regions."
  [system & [params]]
  (http/get (api system) :regions {:query-params params}))

(defn regions-in-country
  "Fetch regions in `country`."
  [system country & [params]]
  (http/get (api system) :regions-in-country country {:query-params params}))

(defn search-autocomplete
  "Search countries, regions and spots."
  [system & [params opts]]
  (http/get (api system) :search-autocomplete (merge opts {:query-params params})))

(defn search-details
  "Search countries, regions and spots."
  [system & [params opts]]
  (http/get (api system) :search-details (merge opts {:query-params params})))

(defn spot
  "Fetch the `spot`."
  [system spot & [params]]
  (http/get (api system) :spot spot {:query-params params}))

(defn spots
  "Fetch spots."
  [system & [params]]
  (http/get (api system) :spots {:query-params params}))

(defn spots-around
  "Fetch the spots around `spot`."
  [system spot & [params]]
  (http/get (api system) :spots-around-spot spot {:query-params params}))

(defn spots-in-region
  "Fetch spots in `region`."
  [system region & [params]]
  (http/get (api system) :spots-in-region region {:query-params params}))

(defn spots-in-country
  "Fetch spots in `country`."
  [system country & [params]]
  (http/get (api system) :spots-in-country country {:query-params params}))

(defn spot-photos
  "Fetch the photos for `spot`."
  [system spot & [params]]
  (http/get (api system) :spot-photos spot {:query-params params}))

(defn spot-weather
  "Fetch the weather for `spot`."
  [system spot & [params]]
  (http/get (api system) :spot-weather spot {:query-params params}))

(defn user
  "Fetch the `user`."
  [system user & [params]]
  (http/get (api system) :user user {:query-params params}))

(defn like-photo
  "Like a photo."
  [system photo & [opts]]
  (http/post (api system) :photo-likes photo opts))

(defn dislike-photo
  "Like a photo."
  [system photo & [opts]]
  (http/delete (api system) :photo-likes photo opts))

(defn wave-heights-chart
  "Return the wave heights chart for `spot`."
  [system spot & [params]]
  (let [url (str (api/url-for (api system) :spot spot) "/wave-heights.svg")]
    (http/get (api system) url {:query-params params})))

(defn validate-user
  "Validate the `user`."
  [system user & [opts]]
  (http/post (:api-client system) :validate-user
             {:transit-params user}))

(defn save-user-settings
  "Save the user settings."
  [system user settings & [params]]
  (let [url (str (api/url-for (api system) :user user) "/settings")]
    (http/put (api system) url {:transit-params settings})))
