(ns burningswell.web.time
  (:refer-clojure :exclude [format])
  (:require [#?(:clj clj-time.core :cljs cljs-time.core)  :as core]
            [#?(:clj clj-time.coerce :cljs cljs-time.coerce)  :as coerce]
            [#?(:clj clj-time.format :cljs cljs-time.format)  :as format]
            #?(:cljs [cljs-time.format :as format])
            [no.en.core :refer [parse-integer]])
  (:import #?(:clj [org.joda.time LocalDate LocalDateTime DateTime DateTimeZone])
           #?(:cljs [goog.date Date DateTime Interval])
           #?(:cljs [goog.i18n DateTimeFormat TimeZone])))

(defn formatter
  "Return a date time formatter for `pattern`."
  [pattern]
  #?(:clj (format/formatter pattern) :cljs (DateTimeFormat. pattern)))

(def date-time-pattern
  "The date and time formatter."
  (formatter "EEE, d MMM, HH:mm"))

(def date-pattern
  "The date formatter."
  (formatter "EEE, d MMM"))

(def time-pattern
  "The time formatter."
  (formatter "HH:mm"))

(def hour-pattern
  "The hour formatter."
  (formatter "HH"))

(defn date-time
  "Return date time instance."
  ([year]
   (date-time year 1 1 0 0 0 0))
  ([year month]
   (date-time year month 1 0 0 0 0))
  ([year month day]
   (date-time year month day 0 0 0 0))
  ([year month day hour]
   (date-time year month day hour 0 0 0))
  ([year month day hour minute]
   (date-time year month day hour minute 0 0))
  ([year month day hour minute second]
   (date-time year month day hour minute second 0))
  ([ year  month  day  hour minute  second  millis]
   #?(:clj (LocalDateTime. year month day hour minute second millis)
      :cljs (DateTime. year (dec month) day hour minute second millis))))

(defn now
  "Return the current date time."
  []
  #?(:clj (LocalDateTime.) :cljs (DateTime.)))

(defn year
  "Return the year of `date-time`."
  [date-time]
  (.getYear date-time))

(defn month
  "Return the year of `date-time`."
  [date-time]
  #?(:clj (.getMonthOfYear date-time)
     :cljs (inc (.getMonth date-time))))

(defn day
  "Return the day of `date-time`."
  [date-time]
  (#?(:clj .getDayOfMonth :cljs .getDate) date-time))

(defn date-time?
  "Return true if `obj` is a time."
  [obj]
  (instance? #?(:clj LocalDateTime :cljs DateTime) obj))

(defn date?
  "Return true if `obj` is a date."
  [obj]
  (instance? #?(:clj LocalDate :cljs Date) obj))

(defn to-date
  "Convert `obj` to a date."
  [obj]
  (cond
    (date? obj)
    obj
    (date-time? obj)
    #?(:clj (.toLocalDate obj)
       :cljs (Date. (.getYear obj) (.getDate obj) (.getDay obj)))))

(defn to-date-time
  "Convert `obj` to a date time."
  [obj]
  #?(:clj (coerce/to-date-time obj)
     :cljs (throw (ex-info "Not implemented yet." {}))))

(defn today
  "Return the goog.date.Date for today."
  []
  (let [now (now)]
    (date-time (year now) (month now) (day now))))

(defn days
  "Return the interval in days."
  [n-days]
  #?(:clj (core/days n-days) :cljs (Interval. 0 0 n-days)))

(defn plus
  "Return the interval in days."
  [date-time interval]
  #?(:clj (core/plus date-time interval)
     :cljs (doto (.clone date-time) (.add interval))))

(defn tomorrow
  "Return the date time object for tomorrow."
  []
  (plus (today) (days 1)))

(defn yesterday
  "Return the date time object for yesterday."
  []
  (plus (today) (days -1)))

(defn format
  "Format `date-time` using `formatter`."
  [formatter date-time & [timezone]]
  #?(:clj
     (cond
       (or (instance? LocalDateTime date-time)
           (instance? LocalDate date-time))
       (format/unparse-local formatter date-time)
       :else
       (format/unparse
        (.withZone formatter (or timezone (DateTimeZone/getDefault)))
        (to-date-time date-time)))
     :cljs (.format formatter date-time timezone)))

(defn format-time
  "Format `date-time` in date time format."
  [date-time & [time-zone]]
  (format time-pattern date-time time-zone))

(defn format-date
  "Format `date-time` in date format."
  [date-time & [time-zone]]
  (format date-pattern date-time time-zone))

(defn same-day?
  "Return true of `date-time-1` and `date-time-2` are the same day,
  otherwise false."
  [date-time-1 date-time-2]
  (and (= (year date-time-1)
          (year date-time-2))
       (= (month date-time-1)
          (month date-time-2))
       (= (day date-time-1)
          (day date-time-2))))

(defn pretty-date [date-time]
  "Format `date` in a pretty format."
  (cond
    (same-day? date-time (yesterday))
    (str "Yesterday")
    (same-day? date-time (today))
    (str "Today")
    (same-day? date-time (tomorrow))
    (str "Tomorrow")
    :else (format date-pattern date-time)))

(defn zone-by-offset
  "Return the time zone for the given offset."
  [offset-hours]
  (when offset-hours
    #?(:clj (DateTimeZone/forOffsetHours offset-hours)
       :cljs (TimeZone.createTimeZone (* -1 (* offset-hours 60))))))

(defn zone-id
  "Return the time zone id."
  [timezone]
  #?(:clj (.getID timezone) :cljs (.getTimeZoneId timezone)))

;; (defn day-time?
;;   "Return true if `time` is at day `timezone`, otherwise false."
;;   [time timezone]
;;   (let [timezone (zone-by-offset (:offset timezone))
;;         hour-str (format hour-pattern time timezone)]
;;     (<= 5 (parse-integer hour-str) 23)))

(defn pretty-time
  "Format `time` in a pretty format."
  [time & [timezone]]
  (let [timezone (zone-by-offset (:offset timezone))]
    (str (pretty-date time) ", " (format time-pattern time timezone))))

(defn query-param
  [date-time]
  (format/unparse
   (format/formatters :date-time)
   (coerce/to-date-time date-time)))
