(ns burningswell.web.components.wave-heights-chart
  (:require [clojure.string :as str]
            [goog.dom.classes :as classes]
            [goog.events.KeyCodes :as KeyCodes]
            [goog.math.Rect]
            [goog.style :as style]
            [om.core :as om]
            [om-tools.core :refer-macros [defcomponent]]
            [sablono.core :refer-macros [html]]))

(defn time-attribute
  "Return the data-time attribute of `element` as a date."
  [element]
  (some-> (.getAttribute element "data-time")
          (js/parseInt)
          (js/Date.)))

(defn select-query
  "Return the child elements of `owner` that match `selector`."
  [owner selector]
  (array-seq (.querySelectorAll (om/get-node owner) selector)))

(defn time-elements
  "Return the child elements of `owner` that have a data-time attribute."
  [owner]
  (select-query owner "g.wave-heights-chart__day"))

(defn parse-timestamps
  "Parse the timestamps of the chart."
  [owner]
  (map time-attribute (time-elements owner)))

(defn within-bounds?
  "Returns true if `x` and `y` are within the bounding box of
  `element`."
  [element x y]
  (let [bounds (style/getBounds element)]
    (.intersection bounds (goog.math.Rect. x y 1 1))))

(defn clear-highlights
  "Clear all highlighted bars in the chart."
  [owner]
  (doseq [node (select-query owner "g.wave-heights-chart__day")]
    (.setAttribute node "class" "wave-heights-chart__day")))

(defn highlight-selector
  "Return the selector for the bar at `time`."
  [time]
  (str "g.wave-heights-chart__day[data-time='" (.getTime time) "']"))

(defn highlight-by-time
  "Highlight the bar in chart by `time`."
  [owner time]
  (doseq [node (select-query owner (highlight-selector time))]
    (->> ["wave-heights-chart__day"
          "wave-heights-chart__day--highlight"]
         (str/join " ")
         (.setAttribute node "class"))))

(defn publish-weather
  "Publish the weather forecast for `time`."
  [owner time]
  (clear-highlights owner)
  (highlight-by-time owner time)
  (om/set-state! owner :timestamp time)
  (when-let [handler (om/get-state owner :on-select)]
    (handler time)))

(defn select-next-time
  "Select the next time in the bar chart."
  [owner]
  (let [{:keys [timestamp timestamps]} (om/get-state owner)]
    (some->> (drop-while #(not= timestamp %) timestamps)
             next first (publish-weather owner))))

(defn select-previous-time
  "Select the previous time in the bar chart."
  [owner]
  (let [{:keys [timestamp timestamps]} (om/get-state owner)]
    (some->> (take-while #(not= timestamp %) timestamps)
             last (publish-weather owner))))

(defn on-key-down
  "Handle key down events."
  [owner event]
  (let [key (.-keyCode event)]
    (cond
      (= KeyCodes/LEFT key)
      (select-previous-time owner)
      (= KeyCodes/RIGHT key)
      (select-next-time owner))))

(defn on-mouse-enter
  "Handle mouse enter events."
  [owner event]
  (.preventDefault event))

(defn on-mouse-over
  "Handle mouse leave events."
  [owner event]
  (some->> (.-parentNode (.-target event))
           (time-attribute)
           (publish-weather owner))
  (.preventDefault event))

(defn on-touch-move
  "Handle touch move events."
  [owner event]
  (let [touches (.-touches event)]
    (when (pos? (.-length touches))
      (let [touch (aget (.-touches event) 0)
            [x y] [(.-pageX touch) (.-pageY touch)]]
        (some->> (time-elements owner)
                 (filter #(within-bounds? % x y))
                 (last)
                 (time-attribute)
                 (publish-weather owner))))
    (.stopPropagation event)))

(defn init-timestamps
  "Initialize state with timestamps from the chart."
  [owner]
  (let [timestamps (parse-timestamps owner)]
    (om/set-state! owner :timestamps (sort timestamps))
    (when-let [timestamp (first timestamps)]
      (om/set-state! owner :timestamp timestamp)
      (highlight-by-time owner timestamp))))

(defcomponent wave-height-chart
  "Return a spot forecast component."
  [chart owner opts]
  (init-state [_]
    {:class (or (:class opts) "wave-height")
     :on-select (:on-select opts)
     :tab-index (or (:tab-index opts) 0)
     :timestamp nil
     :timestamps []})
  (did-mount [_]
    (init-timestamps owner))
  (did-update [this prev-chart prev-state]
    (when (not= prev-chart chart)
      (init-timestamps owner)))
  (render [_]
    (let [{:keys [class tab-index]} (om/get-state owner)]
      (html
       [:div
        {:class class
         :dangerouslySetInnerHTML #js {:__html (or chart "")}
         :on-key-down #(on-key-down owner %)
         :on-mouse-enter #(on-mouse-enter owner %)
         :on-mouse-over #(on-mouse-over owner %)
         :on-touch-move #(on-touch-move owner %)
         :tab-index tab-index}]))))
