(ns burningswell.web.webdriver
  (:require [clj-webdriver.core :as core]
            [clj-webdriver.driver :as driver]
            [clj-webdriver.remote.server :as remote]
            [burningswell.web.taxi :as taxi]
            [clojure.walk :refer [stringify-keys]]
            [com.stuartsierra.component :as component])
  (:import clj_webdriver.driver.Driver
           clj_webdriver.remote.server.RemoteServer
           org.openqa.selenium.remote.RemoteWebDriver))

(def ^:dynamic *driver* nil)

(defn browser-name
  "Reutrn the browser name from `config`."
  [config]
  (or (:browser-name config) "chrome"))

(defn cache-driver
  "Set the dynamic var `*driver*` to `driver`."
  [driver]
  (alter-var-root #'*driver* (constantly driver))
  driver)

(defn configure-driver
  "Configure the `driver`."
  [driver]
  (taxi/implicit-wait driver (:implicit-wait driver))
  driver)

;; Local webdriver

(defn local-driver
  "Return the local webdriver for `config`."
  [config]
  (driver/map->Driver
   {:browser (keyword (browser-name config))
    :implicit-wait (* 1000 10)}))

(defn start-local
  "Start the local webdriver."
  [driver]
  (-> (merge driver (core/new-driver driver))
      (configure-driver)
      (cache-driver)))

(extend-type Driver
  component/Lifecycle
  (start [driver]
    (or *driver* (start-local driver)))
  (stop [driver]
    driver))

;; Remote webdriver

(defn remote-spec
  "Return the remote webdriver for `config`."
  [config]
  (stringify-keys
   {:browserName (browser-name config)
    :build (:travis-job-number config)
    :device-orientation (:device-orientation config)
    :deviceName (:device-name config)
    :name "burningswell"
    :platform (:platform config "OS X 10.11")
    :screen-resolution (:screen-resolution config "1920x1440")
    :tunnel-identifier (or (:travis-job-number config) "burningswell")
    :version (:version config "51.0")}))

(defn remote-host
  "Return the remote host from `config`."
  [config]
  (let [remote (:remote config)]
    (str (:username remote) ":"
         (:access-key remote) "@"
         (:server-name remote))))

(defn remote-driver
  "Return the webdriver config."
  [config]
  (remote/map->RemoteServer
   {:browser-spec (remote-spec config)
    :remote
    {:username (or (:sauce-username config) "burningswell")
     :access-key (:sauce-access-key config)
     :server-name "localhost"
     :server-port 4445}
    :implicit-wait (* 10 1000)}))

(defn remote-job-url
  "Return the job url from `driver`."
  [driver]
  (when (instance? RemoteWebDriver (:webdriver driver))
    (str "https://saucelabs.com/beta/tests" "/"
         (.getSessionId (:webdriver driver)))))

(defn start-remote
  "Return a new remote driver for `config` and `browser-spec`."
  [driver]
  (let [[server started]
        (remote/new-remote-session
         {:existing true
          :host (remote-host driver)
          :port (-> driver :remote :server-port)}
         {:capabilities (:browser-spec driver)})
        started (merge started driver)]
    (println (str "\nRunning tests on SauceLabs: "
                  (remote-job-url started) "\n"))
    (-> (configure-driver started)
        (cache-driver))))

(extend-type RemoteServer
  component/Lifecycle
  (start [driver]
    (or *driver* (start-remote driver)))
  (stop [driver]
    driver))

(defn webdriver
  "Return a new local or remote web driver."
  [config]
  (if (and (:sauce-username config) (:sauce-access-key config))
    (remote-driver config)
    (local-driver config)))
