(ns burningswell.web.components.search
  (:require [burningswell.web.actions :as action]
            [burningswell.web.mixins.event-handler :as events]
            [burningswell.web.router :as router]
            [burningswell.web.util :as util]
            [clojure.string :as str]
            [goog.style :as style]
            [om.core :as om :include-macros true]
            [om-tools.core :refer-macros [defcomponentk defcomponentmethod]]
            [sablono.core :as html :refer-macros [html]]))

(defn on-change
  "Handle change events."
  [owner event]
  (let [query (.-value (.-target event))]
    (om/set-state! owner :show-results true)
    (action/search (om/get-shared owner) query nil)))

(defmulti path-for :type)

(defmethod path-for :country [country]
  (router/path-for :country country))

(defmethod path-for :region [region]
  (router/path-for :region region))

(defmethod path-for :spot [spot]
  (router/path-for :spot spot))

(defmethod path-for :query [query]
  (router/path-for :search (dissoc query :type)))

(defn navigate [system result]
  (if-let [path (path-for result)]
    (action/navigate! system path)))

(defn on-select
  "Handle auto complete selection."
  [data owner index]
  (let [system (om/get-shared owner)
        props (om/get-props owner)
        results (:results props)]
    (om/set-state! owner :show-results false)
    (if-let [result (get results index)]
      (do (navigate system result)
          (om/update! data :text (:term result)))
      (navigate system {:type :query :query (:text props)}))))

(defn on-enter
  "Handle enter events."
  [data owner]
  (let [index (om/get-state owner :index)]
    (on-select data owner index)))

(defn on-touch-start
  "Handle touch start events."
  [data owner index]
  (on-select data owner index))

(defn- set-index! [owner index]
  (om/set-state! owner :index index))

(defn next-result [owner]
  (let [next (inc (om/get-state owner :index))
        results (:results (om/get-props owner))]
    (if (< next (count results))
      (set-index! owner next)
      (set-index! owner -1))))

(defn previous-result [owner]
  (let [prev (dec (om/get-state owner :index))
        results (:results (om/get-props owner))]
    (if (= prev -2)
      (set-index! owner (dec (count results)))
      (set-index! owner prev))))

(defn on-key-down
  "Handle key down events."
  [data owner event]
  (case (.-keyCode event)
    13 (on-enter data owner)
    38 (previous-result owner)
    40 (next-result owner)
    nil))

(defn on-blur
  "Handle blur events."
  [owner event]
  (om/set-state! owner :show-results false))

(defmulti search-result
  (fn [search-result owner] (:type search-result)))

(defcomponentmethod search-result :country
  [data shared]
  (render [_]
    (let [{:keys [continent country]} (:_embedded data)]
      (html
       [:div.search-result
        [:div.search-result__headline
         (util/country-flag country)
         (:term data)]
        [:div.search-result__address
         [:iron-icon {:icon "maps:place"}]
         (:name continent)]]))))

(defcomponentmethod search-result :region
  [data shared]
  (render [_]
    (let [{:keys [continent country]} (:_embedded data)]
      (html
       [:div.search-result
        [:div.search-result__headline
         (util/country-flag country)
         (:term data)]
        [:div.search-result__address
         [:iron-icon {:icon "maps:place"}]
         (:name country) ", " (:name continent)]]))))

(defcomponentmethod search-result :spot
  [data shared]
  (render [_]
    (let [{:keys [continent country region]} (:_embedded data)]
      (html
       [:div.search-result
        [:div.search-result__headline
         (util/country-flag country)
         (:term data)]
        [:div.search-result__address
         [:iron-icon {:icon "maps:place"}]
         (if region (:name region))
         (if region ", ")
         (:name country) ", " (:name continent)]]))))

(defn update-size
  "Update the size of the search result drop down."
  [owner]
  (let [size (style/getSize (om/get-node owner "search"))
        size {:width (.-width size) :height (.-height size)}]
    (om/update-state! owner #(merge % size))))

(defcomponentk search
  [data owner state]
  (:mixins events/handler)
  (init-state [_]
    {:index -1
     :width nil
     :height nil
     :show-results false})
  (did-mount [_]
    (update-size owner)
    (events/listen owner js/window :resize #(update-size owner))
    (let [node (om/get-node owner "search")
          pos (count (:text data))]
      (.setSelectionRange node pos pos)))
  (will-receive-props [this next-props]
    (if-not (= (:text data) (:text next-props))
      (om/set-state! owner :index -1)))
  (render [_]
    (let [{:keys [index width show-results]} @state]
      (html
       [:div.search-input
        [:div.search-input__field
         (html/search-field
          {:auto-complete "off"
           :on-blur #(on-blur owner %)
           :on-change #(on-change owner %)
           :on-key-down #(on-key-down data owner %)
           :placeholder "Search for a country, region or spot"
           :ref "search"}
          "search"
          (:text data))]
        (if show-results
          [:div.search-input__results
           {:style
            {:position "absolute"
             :z-index 1000}}
           (for [[index result] (map-indexed vector (:results data))]
             [:div.search-input__result
              {:class (if (= index (:index @state)) "search-input__result-current")
               :on-touch-start #(on-touch-start data owner index)
               :style {:width width}}
              (om/build search-result result)])])]))))
