(ns burningswell.web.ui.image
  (:require [rum.core :as rum]))

(defn- css-url
  "Return `url` in CSS style."
  [url]
  (if url (str "url(" url ")")))

(defn- attributes
  "Return the attributes of the container element."
  [{:keys [background class height width]}]
  {:class (or class "image")
   :style
   {:background background
    :height height
    :overflow "hidden"
    :width width}})

(defn- sized-element-background-image
  "Return the background image for the sized element."
  [image {:keys [placeholder prevent-load]}]
  (if (and (or placeholder prevent-load)
           (not image))
    (css-url placeholder)
    (if image (css-url (.-src image)))))

(rum/defc sized-element < rum/static
  "Return the sized element."
  [image {:keys [fade sizing] :as opts}]
  [:div
   {:role "img"
    :style
    {:background-image (sized-element-background-image image opts)
     :background-position "50% 50%"
     :background-repeat (if sizing "no-repeat")
     :background-size sizing
     :bottom 0
     :display (if sizing "block" "none")
     :left 0
     :opacity (if image 1 0)
     :position "absolute"
     :right 0
     :top 0
     :transition (if (and image fade) "opacity 0.5s linear")}}])

(defn- image-element-source
  "Return the source attribute of the image element."
  [image {:keys [placeholder preload src]}]
  (cond
    (and placeholder (not image))
    placeholder
    (and preload image)
    (.-src image)
    (not preload)
    src))

(rum/defc image-element < rum/static
  "Return the image element."
  [image {:keys [fade height placeholder preload src sizing width] :as opts}]
  [:img
   {:ref "img"
    :role "none"
    :src (image-element-source image opts)
    :style
    {:display (if sizing "none" "block")
     :opacity (if image 1 0)
     :height (or height "auto")
     :width (or width "auto")
     :transition (if (and image fade) "opacity 0.5s linear")}}])

(rum/defc placeholder-element < rum/static
  "Return the placeholder element."
  [image {:keys [image placeholder position sizing]}]
  (when (and placeholder (not image))
    [:div
     {:ref "placeholder"
      :style
      {:background-color "inherit"
       :background-size sizing
       :background-position (if sizing (or position "center"))
       :background-repeat "no-repeat"
       :background-image (css-url placeholder)
       :width "100%"
       :height "100%"}}]))

(defn- image-url
  "Return the image URL."
  [state]
  (-> state :rum/args first :src))

(defn- load-image
  "Load the image."
  [state]
  #?(:cljs (let [image (js/Image.)]
             (set! (.-onload image) #(reset! (::image state) image))
             (set! (.-src image) (image-url state))
             image)))

(def image-mixin
  "The image mixin."
  {:will-mount
   (fn [state]
     (load-image state)
     state)
   :did-remount
   (fn [old-state new-state]
     (when-not (= (image-url old-state) (image-url new-state))
       (load-image new-state))
     new-state)})

(rum/defcs image < rum/static (rum/local nil ::image) image-mixin
  "Render an image."
  [state opts]
  (let [image (some-> state ::image deref)]
    [:div.image (attributes opts)
     [:div.image__content
      {:style
       {:display "block"
        :height "100%"
        :position "relative"
        :width "100%"}}
      (sized-element image opts)
      (image-element image opts)
      (placeholder-element image opts)]]))
