(ns burningswell.web.util
  (:require #?(:cljs [goog.dom :as gdom])
            #?(:cljs [goog.style :as style])
            [burningswell.transit :as transit]
            [clojure.string :as str]
            [geo.core :as geo]
            [no.en.core :refer [format-url parse-url]]
            [rum.core :as rum]))

(defn read-transit-element
  "Read the inner HTML of `element` as transit data."
  [element]
  #?(:cljs (some-> element
                   (gdom/getElement)
                   (.-innerHTML)
                   (transit/decode))))

(defn bounding-box-query-params
  "Convert a bounding box to query params."
  [bounding-box]
  (when bounding-box
    (let [south-west (:south-west bounding-box)
          north-east (:north-east bounding-box)]
      {:top (geo/point-y north-east)
       :right (geo/point-x north-east)
       :bottom (geo/point-y south-west)
       :left (geo/point-x south-west)})))

(defn location-query-params
  "Convert a location to query params."
  [location]
  (when location
    {:latitude (geo/point-y location)
     :longitude (geo/point-x location)}))

(defn country-flag-url
  "Return the URL of the `country` flag."
  [country]
  (when-let [code (:iso-3166-1-alpha-2 country)]
    (str "/images/countries/flags/4x3/" (str/lower-case code) ".svg")))

(rum/defc country-flag < rum/static
  "Render the `country` flag image."
  [country]
  (when-let [url (country-flag-url country)]
    [:img.country-flag {:src url}]))

(defn viewport-size
  "Return the current viewport size."
  []
  #?(:cljs (let [size (gdom/getViewportSize)]
             {:width (.-width size)
              :height (.-height size)})))

(defn region-geo-image-url
  "Return the URL of the `region` image."
  [system region & [{:keys [color width height]} opts]]
  (let [color (str/replace (or color "#ffffff") "#" "")]
    (-> (assoc (:api-client system) :uri (str "/regions/" (:id region) "/image.png"))
        (assoc :query-params
               {:width (or width 20)
                :height (or height 20)
                :color color})
        (format-url))))

(defn pluralize
  "Returns `singular` if `count` is 1, otherwise `plural`."
  [count singular plural]
  (if (= count 1)
    singular plural))

(defn size-of
  "Returns the size of the DOM `node` as a map with :width and :height keys."
  [node]
  #?(:cljs (if-let [size (style/getSize node)]
             {:width (.-width size)
              :height (.-height size)})))
