(ns burningswell.web.components.star-rating
  "A star rating component."
  (:require [clojure.string :as str]
            [goog.events.KeyCodes :as KeyCodes]
            [om.core :as om]
            [om-tools.core :refer-macros [defcomponent]]
            [sablono.core :refer-macros [defhtml html]]))

(defn active?
  "Return true if the star rating is active, otherwise false."
  [owner]
  (om/get-state owner :active))

(defn star-class
  "Return the CSS class for star `n`."
  [owner n]
  (let [{:keys [class highlight rating]}
        (om/get-state owner)]
    (->> [(str class "__star")
          "x-scope" "iron-icon-0" ;; Needed for icon-icon?
          (if (and highlight (<= n highlight))
            (str class "__star--highlight"))
          (if (and rating (<= n rating))
            (str class "__star--rated"))]
         (remove nil?)
         (str/join " "))))

(defn highlight-rating
  "Highlight the current `rating`."
  [owner event rating]
  (when (active? owner)
    (om/set-state! owner :highlight rating))
  (.preventDefault event))

(defn unhighlight-rating
  "Unhighlight the current `rating`."
  [owner event rating]
  (when (active? owner)
    (om/set-state! owner :highlight nil))
  (.preventDefault event))

(defn submit-rating
  "Submit the `rating`."
  [owner]
  (let [{:keys [on-change rating]} (om/get-state owner)]
    (when on-change (on-change rating))))

(defn on-click
  "Handle on click events on the stars."
  [owner event rating]
  (when (active? owner)
    (if (= rating (om/get-state owner :rating))
      (om/set-state! owner :rating nil)
      (om/set-state! owner :rating rating)))
  (submit-rating owner)
  (.preventDefault event))

(defn decrement-rating
  "Decrements the current rating."
  [owner]
  (let [rating (om/get-state owner :rating)]
    (cond
      (= rating 1)
      (do (om/set-state! owner :rating nil)
          (submit-rating owner))
      (pos? rating)
      (do (om/update-state! owner :rating dec)
          (submit-rating owner)))))

(defn increment-rating
  "Increments the current rating."
  [owner]
  (let [{:keys [rating stars]} (om/get-state owner)]
    (when (or (nil? rating) (< rating stars))
      (om/update-state! owner :rating inc)
      (submit-rating owner))))

(defn on-key-down
  "Handle key down events."
  [owner event]
  (when (active? owner)
    (let [key (.-keyCode event)]
      (cond
        (= KeyCodes/LEFT key)
        (decrement-rating owner)
        (= KeyCodes/RIGHT key)
        (increment-rating owner)))))

(defhtml star
  "Render the star `rating`."
  [owner rating]
  [:iron-icon
   {:class (star-class owner rating)
    :icon "star"
    :on-click #(on-click owner % rating)
    :on-mouse-enter #(highlight-rating owner % rating)
    :on-mouse-leave #(unhighlight-rating owner % rating)}])

(defcomponent star-rating
  "A star rating component."
  [rating owner opts]
  (init-state [_]
    {:active (:active opts true)
     :class (or (:class opts) "star-rating")
     :on-change (:on-change opts)
     :rating (:rating opts)
     :stars 5
     :tab-index (or (:tab-index opts) 1)})
  (will-receive-props [this next-rating]
    (om/set-state! owner :rating next-rating))
  (render [_]
    (let [{:keys [active class tab-index stars]}
          (om/get-state owner)]
      (html
       [:div
        {:class class
         :on-key-down #(on-key-down owner %)
         :tab-index (and active tab-index)}
        (for [n (range 1 (inc stars))]
          (star owner n))]))))
