(ns burningswell.web.core
  (:require [burningswell.web.actions :as action]
            [geo.core :as geo]
            [goog.style :as style]
            [no.en.core :refer [format-url parse-url]]
            [om.core :as om :include-macros true]
            [om-tools.core :refer-macros [defcomponentk]]
            [sablono.core :as html :refer-macros [html]]
            [schema.core :as s]))

(extend-type om.core/MapCursor
  geo.core/IPoint
  (point-x [cursor]
    (some-> cursor deref geo/point-x))
  (point-y [cursor]
    (some-> cursor deref geo/point-y)))

(defn point->lat-lng
  "Convert a geo point to a google.maps.LatLng."
  [point]
  (google.maps.LatLng.
   (geo/point-y point)
   (geo/point-x point)))

(defn lat-lng->point
  "Convert a google.maps.LatLng to a geo point."
  [lat-lng]
  (geo/point 4326 (.lng lat-lng) (.lat lat-lng)))

(defn link-to
  "Link to `url`."
  [system url & content]
  (html
   [:a {:href url
        :on-click
        (fn [event]
          (action/navigate! system url)
          (.preventDefault event))}
    content]))

(defn size
  "Return the size of `node` as a map."
  [node]
  (let [size (style/getSize node)]
    {:width (.-width size)
     :height (.-height size)}))

(defn image-resize-url
  "Return the URL for a version the image `url` resized to `size`."
  [url size]
  (if-let [spec (parse-url url)]
    (->> (assoc size :op "resize")
         (assoc spec :query-params )
         (format-url))))

(defcomponentk image
  "A background image component."
  [data :- s/Str owner shared state]
  (did-mount [_]
    (let [size (size (om/get-node owner))
          position (str "0px 0px")]
      (om/set-state! owner :url data)
      (om/set-state! owner :position position)))
  (render [_]
    (let [{:keys [url position]} @state]
      (html
       [:div.bs-image
        {:style
         {:background-image
          (if url (str "url(" url ")"))
          :background-size "cover"
          :background-position position
          :height "100%"}}]))))

(defcomponentk google-map
  "Return a country card component."
  [data owner shared opts]
  (did-mount [_]
    (when-let [node (om/get-node owner "map")]
      (let [location (:location data)
            opts {:center
                  {:lat (geo/point-y location)
                   :lng (geo/point-x location)}
                  :zoom (or (:zoom opts) 4)}
            google-map (google.maps.Map. node (clj->js opts))]
        (om/set-state! owner :map google-map))))
  (render [_]
    (html
     [:div.bs-google-map {:ref "map" :style {:height "100%"}}])))
