(ns burningswell.web.test
  (:require [burningswell.api.system :as api]
            [burningswell.config.core :as config]
            [burningswell.config.test :as test-config]
            [burningswell.system :refer [defsystem]]
            [burningswell.web.cookies :as cookies]
            [burningswell.web.system.client :as client]
            [burningswell.web.system.server :as server]
            [burningswell.web.webdriver :as webdriver]
            [clj-webdriver.taxi :as taxi]
            [com.stuartsierra.component :as component]
            [environ.core :as environ]
            [no.en.core :refer [format-url parse-url]]
            [request.backend.httpkit])
  (:import java.util.regex.Pattern))

(def credentials
  "The credentials of test users."
  {:bodhi {:login "bodhi" :password "secret"}})

(def env
  "The test env."
  test-config/env)

(defn config
  "Return the test config."
  [& [opts]]
  (config/web (merge env opts)))

(defsystem webdriver-system
  "The Burning Swell webdriver system."
  [env]
  (let [env (merge environ/env env)]
    (component/system-map
     :api (api/new-system (config/api env))
     :web (server/server (config/web env))
     :web-client (config/web-client env)
     :webdriver (webdriver/webdriver env))))

(defsystem headless-system
  "The Burning Swell headless system."
  [env]
  (-> (client/client (config/web env))
      (assoc :api (api/new-system (config/api env)))))

(defn browse-to [system path & [params]]
  (let [url (assoc (:web-client system) :uri path :query-params params)]
    (taxi/to (:webdriver system) (format-url url))))

(defn current-path
  "Return the current path."
  [webdriver]
  (some-> (taxi/current-url webdriver)
          parse-url :uri))

(defn wait-for
  [f & [{:keys [sleep timeout]}]]
  (let [sleep (or sleep 500)
        timeout (or timeout (* 5 1000))
        started-at (System/currentTimeMillis)]
    (loop []
      (if-let [result (f)]
        result
        (if (< (System/currentTimeMillis)
               (+ started-at timeout))
          (do (Thread/sleep sleep)
              (recur))
          (throw (ex-info
                  "Wait timeout"
                  {:sleep sleep
                   :timeout timeout})))))))

(defn wait-for-path
  "Wait until the browser has navigated to `path`."
  [webdriver path & [timeout interval]]
  (taxi/wait-until
   webdriver
   (fn [_]
     (if (instance? Pattern path)
       (re-matches path (str (current-path webdriver)))
       (= path (current-path webdriver))))
   (or timeout 10000)
   (or interval 200))
  (str (current-path webdriver)))

(defn find-text
  "Return the region headline."
  [webdriver query]
  (some->> (taxi/find-element webdriver query)
           (taxi/text webdriver)))

(defn find-texts
  "Return the region headline."
  [webdriver query]
  (->> (taxi/find-elements webdriver query)
       (map #(taxi/text webdriver %))))

(defn country-headline
  "Return the country headline."
  [webdriver]
  (find-text webdriver {:css ".country__headline"}))

(defn country-headline-names
  "Return the country name from the headline."
  [webdriver]
  (find-texts webdriver {:css ".country-headline__name"}))

(defn spot-headline
  "Return the spot headline."
  [webdriver]
  (find-text webdriver {:css ".spot-headline"}))

(defn spot-headline-name
  "Return the spot name in the headline."
  [webdriver]
  (find-text webdriver {:css ".spot-headline__name"}))

(defn spot-headline-names
  "Return the spot name from the headline."
  [webdriver]
  (find-texts webdriver {:css ".spot-headline__name"}))

(defn region-headline
  "Return the region headline."
  [webdriver]
  (find-text webdriver {:css ".region__headline"}))

(defn region-headline-names
  "Return the region name from the headline."
  [webdriver]
  (find-texts webdriver {:css ".region-headline__name"}))

(defn current-layout-link-text
  "Return the text from the current layout link."
  [webdriver]
  (find-text webdriver {:css ".layout__link--current .layout__link_text"}))

(defn current-layout-link-icon
  "Return the text from the current layout link."
  [webdriver]
  (find-text webdriver {:css ".layout__link--current .layout__link_icon"}))

(defn set-location-cookie!
  "Set the location cookie."
  [webdriver location]
  (taxi/add-cookie webdriver {:name "location" :value (cookies/encode location) :path "/"}))
