(ns burningswell.web.geo-location
  (:require [burningswell.web.bus :as bus]
            [burningswell.web.logging :as log]
            [com.stuartsierra.component :as component]
            [geo.core :as geo]))

(def logger
  "The logger of the current namespace."
  (log/logger "burningswell.web.geo-location"))

(defn position->point
  "Convert `position` into a geo point."
  [position]
  (let [coords (.-coords position)]
    (geo/point 4326 (.-longitude coords) (.-latitude coords))))

(defn log-current-location [location]
  "Log the current geo location."
  [location]
  (->> (str "Current geo location: "
            (geo/point-y location) ", "
            (geo/point-x location))
       (log/info logger)))

(defn publish
  "Publish the geo location."
  [geo-location location]
  (log-current-location location)
  (reset! (:location geo-location) location)
  (bus/publish (:bus geo-location) ::location {:location location}))

(defn start-watch
  "Start geo location watching."
  [geo-location]
  (assoc geo-location :listener
         #?(:clj nil
            :cljs (.watchPosition
                   (.-geolocation js/navigator)
                   (fn [position]
                     (publish geo-location (position->point position)))
                   (fn [error]
                     (->> (str "Can't get geo location: " (.-message error) " "
                               "(" (.-code error) ").")
                          (log/warning logger)))))))

(defn clear-watch
  "Stop geo location watching."
  [geo-location]
  (when-let [listener (:listener geo-location)]
    #?(:cljs (.clearWatch (.-geolocation js/navigator) listener)))
  (assoc geo-location :listener nil))

(defn current-location
  "Return the current location."
  [geo-location]
  (-> geo-location :location deref))

(defn start-geo-location
  "Start the geo location component."
  [geo-location]
  (if (or #?(:cljs (nil? (.-geolocation js/navigator)))
          (:listener geo-location))
    geo-location
    (let [geo-location (start-watch geo-location)]
      (log/info logger "Geo location successfully started.")
      geo-location)))

(defn stop-geo-location
  "Stop the geo location component."
  [geo-location]
  (let [geo-location (clear-watch geo-location)]
    (log/info logger "Geo location successfully stopped.")
    geo-location))

(defrecord GeoLocation [listener location]
  component/Lifecycle
  (start [geo-location]
    (start-geo-location geo-location))
  (stop [geo-location]
    (stop-geo-location geo-location)))

(defn geo-location
  "Return a new geo location component."
  [& [opts]]
  (-> (assoc opts :location (atom nil))
      (map->GeoLocation)
      (component/using [:bus])))

;; (ns burningswell.web.geo-location
;;   (:require [coolant.core :as coolant]
;;             [geo.core :as geo]
;;             [com.stuartsierra.component :as component]))

;; (defn geocoder
;;   "Return the geocoder."
;;   []
;;   (.-geolocation js/navigator))

;; (defn coords->point
;;   "Convert geo location coordinates to a point."
;;   [coords]
;;   (geo/point 4326 (.-longitude coords) (.-latitude coords)))

;; (defn set-position!
;;   "Set the current geo position."
;;   [geo-location position]
;;   (let [location (coords->point (.-coords position))]
;;     ;; (swap! (:state geo-location) assoc :location location)
;;     (coolant/dispatch! (:store geo-location) (coolant/message :location/changed location))
;;     (assoc! (:local-storage geo-location) :location location)))

;; (defn get-current
;;   [geo-location]
;;   (.getCurrentPosition
;;    (geocoder)
;;    (fn [position]
;;      (set-position! geo-location position))
;;    (fn [error]
;;      (.log js/console "Can't get geo location.")
;;      (prn error))))

;; (defn start-watch
;;   "Start geo location watching."
;;   [geo-location]
;;   (assoc geo-location :listener
;;          (.watchPosition
;;           (geocoder)
;;           (fn [position]
;;             (set-position! geo-location position))
;;           (fn [error]
;;             (.log js/console "Can't get geo location.")
;;             (prn error)))))

;; (defn clear-watch
;;   "Stop geo location watching."
;;   [geo-location]
;;   (when-let [listener (:listener geo-location)]
;;     (.clearWatch (geocoder) listener))
;;   (assoc geo-location :listener nil))

;; (defn start-geo-location
;;   "Start the geo location component."
;;   [geo-location]
;;   (if (or (:listener geo-location)
;;           (nil? (geocoder)))
;;     geo-location
;;     (do (get-current geo-location)
;;         (start-watch geo-location))))

;; (defn stop-geo-location
;;   "Stop the geo location component."
;;   [geo-location]
;;   (clear-watch geo-location))

;; (defrecord GeoLocation [listener state]
;;   component/Lifecycle
;;   (start [geo-location]
;;     (start-geo-location geo-location))
;;   (stop [geo-location]
;;     (stop-geo-location geo-location)))

;; (defn new-geo-location
;;   "Return a new geo location component."
;;   []
;;   (map->GeoLocation {}))
