(ns burningswell.web.components.rate-session
  (:require-macros [cljs.core.async.macros :refer [go]])
  (:require [burningswell.web.actions :as action]
            [burningswell.web.components.paper-button :refer [->paper-button]]
            [burningswell.web.components.star-rating :refer [->star-rating]]
            [burningswell.web.components.weather :refer [->weather]]
            [burningswell.web.mixins.shake :as shake]
            [burningswell.web.mixins.event-handler :as events]
            [om-tools.core :refer-macros [defcomponent]]
            [om.core :as om]
            [sablono.core :refer-macros [defhtml html]]))

(defn open-dialog
  "Open the rate session dialog."
  [owner]
  (.open (om/get-node owner)))

(defn close-dialog
  "Close the rate session dialog."
  [owner]
  (.close (om/get-node owner)))

(defhtml current-rating [data owner]
  [:div.rate-session__current-rating
   {:on-click #(open-dialog owner)}
   (->star-rating data)])

(defhtml previous-rating [data owner]
  [:div.rate-session__previous-rating
   (->star-rating data)])

(defn process-events
  "Process events from the channel."
  [owner]
  (if-let [channel (om/get-state owner :channel)]
    (go (loop [[cmd & args] (<! channel)]
          (when cmd
            (case cmd
              :close (close-dialog owner)
              :open (open-dialog owner))
            (recur (<! channel)))))))

(defn open-dialog?
  "Returns true if the dialog is open, otherwise false."
  [session]
  (:open session))

(defn on-time-change
  "Handle time display change events."
  [owner time]
  (action/change-session-time (om/get-shared owner) time))

(defn on-dialog-close
  "Handle dialog close events."
  [owner]
  ;; TODO: Pass spot?
  (action/close-session-dialog (om/get-shared owner) nil))

(defn on-submit
  "Handle form submit events."
  [owner]
  (let [session (om/get-props owner)]
    (action/submit-session
     (om/get-shared owner)
     {:spot (:spot session)
      :session
      {:spot-id (-> session :spot :id)
       :started-at (js/Date.)
       :stopped-at (js/Date.)
       :rating (:rating session)}})))

(defn on-rating-changed
  "Handle rating change events."
  [owner rating]
  (action/change-session-rating (om/get-shared owner) rating))

(defhtml action-buttons
  "Render the action buttons."
  [owner]
  (let [class (om/get-state owner :class)]
    [:div {:class (str class "__actions")}
     (->paper-button
      "Submit"
      {:opts
       {:class (str class "__submit")
        :on-click #(on-submit owner)}})
     (->paper-button
      "Cancel"
      {:opts
       {:class (str class "__cancel")
        :on-click #(close-dialog owner)}})]))

(defhtml description
  "Render the rate session description."
  [owner session]
  (let [class (om/get-state owner :class)]
    [:p {:class (str class "__description")}
     "Submit a rating for one of your surf sessions at "
     [:span {:class (str class "__spot-name")}
      (-> session :spot :name)] " in "
     [:span {:class (str class "__country-name")}
      (-> session :spot :_embedded :country :name)]
     ". Your rating will be used to improve the personalized surf
     forecasts for you and other people."]))

(defn shake?
  "Return true if the dialog should shake, otherwise false."
  [owner next-session]
  (and (:failed-at next-session)
       (not= (:failed-at next-session)
             (:failed-at (om/get-props owner)))))

(defcomponent rate-session
  [session owner opts]
  (:mixins events/handler shake/shake)
  (init-state [_]
    {:channel (:channel opts)
     :class (or (:class opts) "rate-session")
     :on-submit (:on-submit opts)})
  (will-mount [_]
    (process-events owner))
  (did-mount [_]
    (let [node (om/get-node owner)]
      ;; TODO: How to fix z-indes
      ;; (.setAttribute node "modal" true)
      (events/listen owner node :iron-overlay-closed #(on-dialog-close owner)))
    (when (open-dialog? session)
      (open-dialog owner)))
  (will-receive-props [this next-session]
    (if (open-dialog? next-session)
      (open-dialog owner)
      (close-dialog owner))
    (when (shake? owner next-session)
      (shake/shake! owner)))
  (render [_]
    (let [class (om/get-state owner :class)]
      (html
       [:paper-dialog
        {:class class}
        [:div {:class (str class "__content")}
         [:div {:class (str class "__headline")}
          [:h2 "Rate your surf session"]
          (->star-rating
           (:rating session)
           {:opts
            {:class (str class "__rating")
             :on-change #(on-rating-changed owner %)}})]
         (when-let [error (-> session :errors :rating)]
           [:div {:class (str class "__rating-error")} error])
         (description owner session)
         (when (and (:wave-heights session) (:weather session))
           (->weather
            session
            {:opts
             {:class (str class "__weather")
              :on-time-change #(on-time-change owner %)}}))
         (action-buttons owner)]]))))
