(ns burningswell.weather.forecast
  "Create the Burning Swell weather forecast from the raster data."
  (:gen-class)
  (:refer-clojure :exclude [distinct group-by replace update])
  (:require [burningswell.config.core :as config]
            [burningswell.db.connection :as db]
            [burningswell.db.weather :as weather]
            [burningswell.system :refer [defsystem]]
            [burningswell.time :refer [current-duration-str]]
            [clj-time.coerce :refer [to-date-time]]
            [clj-time.core :as time]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [commandline.core :refer [print-help with-commandline]]
            [datumbazo.core :refer :all]
            [environ.core :refer [env]]
            [no.en.core :refer [parse-long]]))

(defsystem system
  "The weather forecast CLI system."
  [{:keys [db]}]
  (component/system-map :db (db/new-db db)))

(defn- parse-spots
  "Parse the command line spot ids."
  [spots]
  (->> (map parse-long spots)
       (remove nil?)
       (map #(hash-map :id %))))

(defn select-start
  "Select the weather forecast start time."
  [db]
  (-> @(select db [(as '(max :reference-time) :reference-time)]
               (from :weather.spot-forecasts))
      first :reference-time to-date-time))

(defn select-end
  "Select the weather forecast end time."
  [db]
  (-> @(select db [(as '(max :valid-time) :valid-time)]
               (from :weather.datasets))
      first :valid-time to-date-time))

(defn- select-forecast-count
  "Select the number of forecast rows."
  [db]
  (-> @(select db [(as '(count :*) :count)]
               (from :weather.spot-forecasts))
      first :count))

(defn- spot-count
  "Return the spot count, either from the `opts` or the `db`."
  [db {:keys [spots]}]
  (if (empty? spots)
    (-> @(select db [(as '(count :*) :count)]
                 (from :spots))
        first :count)
    (count spots)))

(defn import-weather-forecast
  "Create the weather forecasts for spots."
  [db opts]
  (when-not (:skip-import opts)
    (let [started-at (time/now)]
      (log/info "Building weather forecast ...")
      (log/infof "  Start time ....... %s" (:start opts))
      (log/infof "  End time ......... %s" (:end opts))
      (log/infof "  Number of spots .. %s" (spot-count db opts))
      (let [result @(weather/import-spot-forecasts db opts)
            row-count (-> result first :count)]
        (log/infof "  Rows imported .... %s" row-count)
        (log/infof "  Duration ......... %s"
                   (current-duration-str started-at))
        (->> {:import-started-at started-at
              :import-finished-at (time/now)
              :import-row-count row-count}
             (merge opts))))))

(defn refresh-weather-forecast
  "Update the weather forecasts views."
  [db opts]
  (when-not (:skip-refresh opts)
    (let [started-at (time/now)]
      (log/info "Updating weather forecast views ...")
      (let [result @(weather/refresh-3-hourly-spot-weather db)
            row-count (select-forecast-count db)]
        (log/infof "  Rows refreshed ... %s" row-count)
        (log/infof "  Duration ......... %s"
                   (current-duration-str started-at))
        (->> {:refresh-started-at started-at
              :refresh-finished-at (time/now)
              :refresh-row-count row-count}
             (merge opts))))))

(defn run-cli
  "Create the Burning Swell weather forecast from the raster data."
  [{:keys [db]} & [{:keys [spots start end] :as opts}]]
  (with-connection [db db]
    (let [spots (parse-spots spots)
          start (or start
                    (select-start db)
                    (weather/min-dataset-valid-time db))
          end (or end (select-end db) start)]
      (when (or (and start end (time/after? end start))
                (:force opts))
        (->> (assoc opts
                    :end end
                    :spots spots
                    :start start)
             (import-weather-forecast db)
             (refresh-weather-forecast db))))))

(defn -main
  "Create the Burning Swell weather forecast from the raster data."
  [& args]
  (with-commandline [[opts spots] args]
    [[I skip-import "Skip the import of the weather forecast."]
     [R skip-refresh "Skip the refreshing of the weather forecast."]
     [e end "The weather forecast start time." :string "END"]
     [f force "Force import and refresh of the weather forecast."]
     [h help "Print this help."]
     [s start "The weather forecast end time." :time "START"]]
    (if (:help opts)
      (print-help "weather-forecast [OPTION...] [SPOT-ID,...]")
      (with-system [system {:db (config/db env)}]
        (run-cli system (assoc opts :spots spots))))))

(comment (-main "1"))
