(ns burningswell.transit
  (:require #?(:clj [clj-time.coerce :refer [to-long]])
            [cognitect.transit :as transit]
            [geo.transit :as geo-transit])
  #?(:clj (:import [java.io ByteArrayInputStream ByteArrayOutputStream]
                   [java.net URL URI]
                   [java.util UUID]
                   [org.joda.time DateTime])))

#?(:clj (def read-uri
          "Read a java.net.URI instance."
          (transit/read-handler
           (fn [uri] (URI. uri)))))

#?(:clj (def read-url
          "Read a java.net.URL instance."
          (transit/read-handler
           (fn [url] (URL. url)))))

#?(:clj (def read-uuid
          "Read a java.util.UUID instance."
          (transit/read-handler
           (fn [uuid] (UUID/fromString uuid)))))

#?(:clj (def write-class
          "Write a Class as a symbol."
          (transit/write-handler
           (constantly "$")
           (fn [class] (str class))
           (constantly nil))))

#?(:clj (def write-date-time
          "Write org.joda.time.DateTime."
          (transit/write-handler
           (constantly "m")
           (fn [date-time]
             (str (to-long date-time)))
           (constantly nil))))

#?(:clj (def write-uri
          "Write a java.net.URI instance."
          (transit/write-handler
           (constantly "burningswell/uri")
           (fn [url] (str url))
           (constantly nil))))

#?(:clj (def write-url
          "Write a java.net.URL instance."
          (transit/write-handler
           (constantly "burningswell/url")
           (fn [url] (str url))
           (constantly nil))))

#?(:clj (def write-uuid
          "Write a java.util.UUID instance."
          (transit/write-handler
           (constantly "burningswell/uuid")
           (fn [url] (str url))
           (constantly nil))))

(def read-handlers
  "The Transit read handlers."
  (merge
   geo-transit/read-handlers
   #?(:clj {"burningswell/uri" read-uri
            "burningswell/url" read-url
            "burningswell/uuid" read-uuid})))

(def write-handlers
  "The Transit write handlers."
  (merge
   geo-transit/write-handlers
   #?(:clj {Class write-class
            DateTime write-date-time
            URI write-uri
            URL write-url
            UUID write-uuid})))

(defn encode
  "Encode `payload` into Transit format."
  [payload & [opts]]
  (let [opts (merge {:handlers write-handlers} opts)]
    #?(:clj (let [format (or (:format opts) :json)
                  stream (ByteArrayOutputStream.)
                  writer (transit/writer stream format opts)
                  _ (transit/write writer payload)]
              (case format
                :json (.toString stream)
                :msgpack (.toByteArray stream)))
       :cljs (transit/write (transit/writer :json opts) payload))))

(defn decode
  "Decode `payload` from Transit format."
  [payload & [opts]]
  (let [opts (merge {:handlers read-handlers} opts)]
    #?(:clj (let [format (or (:format opts) :json)
                  stream (ByteArrayInputStream.
                          (case format
                            :json (.getBytes payload)
                            :msgpack payload))
                  reader (transit/reader stream format opts)]
              (transit/read reader))
       :cljs (transit/read (transit/reader :json opts) payload))))
