(ns burningswell.streams.users.profile-photos
  (:require [burningswell.db.emails :as emails]
            [burningswell.db.photos :as db-photos]
            [burningswell.db.users :as users]
            [burningswell.services.facebook :as facebook]
            [burningswell.services.gravatar :as gravatar]
            [burningswell.services.photos :as photos]
            [burningswell.streams.api.events :as events]
            [burningswell.streams.core :as k]
            [clojure.string :as str]
            [peripheral.core :refer [defcomponent]]
            [clojure.tools.logging :as log]))

(defn config [& [opts]]
  (->> {:application.id "user-profile-photos"
        :input {:events "burningswell.api.events"}
        :output {:created "burningswell.users.profile-photos.created"}}
       (merge opts)))

(defn- events [env builder]
  (.stream builder (-> env :config :input :events)))

(defn- user-created
  "Returns a stream of user-created event."
  [events]
  (events/by-name events :burningswell.api.events/user-created))

(defn- oauth-callback-succeeded
  "Returns a stream of OAuth callback event."
  [events]
  (events/by-name events :burningswell.api.events.oauth/callback-succeeded))

(defn- photo-title
  "Returns the profile photo title for `provider`."
  [provider]
  (str (str/capitalize (name provider)) " Profile Photo"))

(defn gravatar-photos
  "Save the Gravatar profile photos for a create user `event`."
  [{:keys [db gravatar photos] :as env} event]
  (when-let [user (users/by-id db (:user-id event))]
    (for [email (emails/by-user db user)
          :let [{:keys [thumbnail-url]} (gravatar/profile gravatar email)]
          :when (empty? (db-photos/by-user-and-url db user thumbnail-url))]
      {:title (photo-title :gravatar)
       :url thumbnail-url
       :user-id (:id user)})))

(defmulti profile-photo-urls
  "Returns the profile photo urls from an OAuth callback event."
  (fn [env event] (-> event :provider keyword)))

(defmethod profile-photo-urls :facebook
  [{:keys [db facebook]} {:keys [user-id profile]}]
  (let [response (facebook/profile-photo facebook profile)]
    (when (not (:is-silhouette response))
      (-> response :url vector))))

(defmethod profile-photo-urls :google [_ event]
  (when (not (-> event :profile :image :is-default))
    (some-> event :profile :image :url vector)))

(defmethod profile-photo-urls :linkedin [_ event]
  (some-> event :profile :picture-url vector))

(defmethod profile-photo-urls :twitter [_ {:keys [profile]}]
  (when (not (:default-profile-image profile))
    (some-> (or (:profile-image-url-https profile)
                (:profile-image-url-http profile))
            vector)))

(defn oauth-provider-photos
  "Returns the profile photos from an OAuth callback event."
  [{:keys [db photos] :as env} {:keys [provider] :as event}]
  (when-let [user (users/by-id db (:user-id event))]
    (for [url (profile-photo-urls env event)]
      (when (empty? (db-photos/by-user-and-url db user url))
        {:title (photo-title provider)
         :url url
         :user-id (:id user)}))))

(defn save-profile-photo!
  "Save the profile `photo` to the database."
  [{:keys [db photos] :as env} photo]
  (try (when-let [user (users/by-id db (:user-id photo))]
         (let [photo (photos/create-profile-photo! photos user photo)]
           (log/infof "Saved profile photo for user %s: %s."
                      (:user-id photo) (:url photo))
           photo))
       (catch Throwable e
         (case (-> e ex-data :type)
           ::photos/download-error nil
           (throw e)))))

(defn make-topology
  [{:keys [gravatar] :as env}]
  (k/with-build-stream builder
    (let [events (events env builder)]
      (-> (user-created events)
          (k/flat-map-vals #(gravatar-photos env %))
          (k/map-vals #(save-profile-photo! env %))
          (.to (-> env :config :output :created)))
      (-> (oauth-callback-succeeded events)
          (k/flat-map-vals #(oauth-provider-photos env %))
          (k/map-vals #(save-profile-photo! env %))
          (.to (-> env :config :output :created))))))

(defcomponent ProfilePhotos [config]
  :this/as *this*
  :topology (make-topology *this*)
  :stream (k/start-topology (k/props config) topology) #(.close %))

(defn profile-photos [& [opts]]
  (map->ProfilePhotos {:config (config opts)}))
