(ns burningswell.streams.route-views
  (:require [burningswell.streams.core :as k]
            [no.en.core :refer [parse-url]]
            [burningswell.routes :as routes]
            [peripheral.core :refer [defcomponent]])
  (:import [org.apache.kafka.streams KafkaStreams StreamsBuilder StreamsConfig]
           org.apache.kafka.streams.KeyValue))

(defn total-config [& [opts]]
  (->> {:application.id "route-views-total"
        :input "burningswell.page.views.total"
        :output "burningswell.page.views.total-per-user"}
       (merge opts)))

(defn total-per-user-config [& [opts]]
  (->> {:application.id "route-views-total-per-user"
        :input "burningswell.page.views.total-per-user"
        :output "burningswell.route.views.total-per-user"}
       (merge opts)))

(defn- start-stream [props topology]
  (doto (KafkaStreams. topology (StreamsConfig. props))
    (.cleanUp)
    (.start)))

(defn- match-route [page]
  (some-> page :url parse-url :uri routes/match (dissoc :uri)))

(defn total-topology [env config]
  (k/with-build-stream builder
    (-> (.stream builder (:input config))
        (.map (k/kv-mapper [page views]
                (KeyValue. (match-route page) views)))
        (k/filter (fn [route views]
                    (some? (:handler route))))
        (.through (:output config)))))

(defn total-per-user-topology [env config]
  (k/with-build-stream builder
    (-> (.stream builder (:input config))
        (.map (k/kv-mapper [page views]
                (KeyValue.
                 (some-> (match-route page)
                         (merge (select-keys page [:user-id])))
                 views)))
        (k/filter (fn [route views]
                    (some? (:handler route))))
        (.through (:output config)))))

(defcomponent Total [config]
  :this/as *this*
  :props (k/props config)
  :topology (total-topology *this* (total-config ))
  :stream (start-stream props topology) #(.close %))

(defn total [config]
  (map->Total {:config config}))

(defcomponent TotalByUser [config]
  :this/as *this*
  :props (k/props config)
  :topology (total-per-user-topology *this* (total-per-user-config))
  :stream (start-stream props topology) #(.close %))

(defn total-per-user [config]
  (map->TotalByUser {:config config}))
