(ns burningswell.streams.core
  (:refer-clojure :exclude [filter map group-by])
  (:import burningswell.kafka.serdes.transit
           java.util.UUID
           org.apache.kafka.clients.consumer.ConsumerConfig
           [org.apache.kafka.streams KafkaStreams KeyValue StreamsBuilder StreamsConfig]
           [org.apache.kafka.streams.kstream KeyValueMapper Predicate Reducer ValueMapper]))

(defn props [& [config]]
  (println "Using bootstrap servers:" (:bootstrap.servers config))
  {ConsumerConfig/AUTO_OFFSET_RESET_CONFIG, "earliest"
   StreamsConfig/APPLICATION_ID_CONFIG (or (:application.id config) (str (UUID/randomUUID)))
   StreamsConfig/BOOTSTRAP_SERVERS_CONFIG (or (:bootstrap.servers config) "localhost:9092")
   StreamsConfig/CACHE_MAX_BYTES_BUFFERING_CONFIG 0
   StreamsConfig/DEFAULT_KEY_SERDE_CLASS_CONFIG (.getName transit)
   StreamsConfig/DEFAULT_VALUE_SERDE_CLASS_CONFIG (.getName transit)
   StreamsConfig/NUM_STREAM_THREADS_CONFIG (str (or (:num.stream.threads config) 1))})

(defn config [& [opts]]
  (StreamsConfig. (props opts)))

(defmacro reducer
  {:style/indent 1}
  [kv & body]
  `(reify Reducer
     (apply [_# ~(first kv) ~(second kv)]
       ~@body)))

(defmacro value-mapper
  {:style/indent 1}
  [kv & body]
  `(reify ValueMapper
     (apply [_# ~(first kv)]
       ~@body)))

(defmacro kv-mapper
  {:style/indent 1}
  [kv & body]
  `(reify KeyValueMapper
     (apply [_# ~(first kv) ~(second kv)]
       ~@body)))

(defmacro pred
  {:style/indent 1}
  [kv & body]
  `(reify Predicate
     (test [_# ~(first kv) ~(second kv)]
       ~@body)))

(defn start-topology [props topology]
  (prn props)
  (doto (KafkaStreams. topology (StreamsConfig. props))
    (.cleanUp)
    (.start)))

(defmacro with-build-stream [builder-sym & body]
  `(let [~builder-sym (StreamsBuilder.)]
     ~@body
     (.build ~builder-sym)))

(defn flat-map [s kv-fn]
  (.flatMap s (kv-mapper [k v] (kv-fn k v))))

(defn flat-map-vals [s val-fn]
  (.flatMapValues s (value-mapper [v] (or (val-fn v) []))))

(defn filter [s kv-fn]
  (.filter s (pred [key value] (kv-fn key value))))

(defn group-by [s kv-fn]
  (.groupBy s (kv-mapper [key value] (kv-fn key value))))

(defn map [s kv-fn]
  (.map s (kv-mapper [key value]
            (let [[k v] (kv-fn key value)]
              (KeyValue. k v)))))

(defn map-vals [s val-fn]
  (.mapValues s (value-mapper [v] (val-fn v))))

(defn random-uuid []
  (java.util.UUID/randomUUID))
