(ns burningswell.services.photos
  (:require [burningswell.db.images :as images]
            [burningswell.db.photos :as photos]
            [burningswell.db.users :as users]
            [burningswell.services.storage :as storage]
            [clj-http.client :as http]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [datumbazo.core :as sql]
            [image-resizer.core :as resizer]
            [pandect.core :as digest]
            [ring.util.codec :refer [base64-encode]])
  (:import java.awt.image.BufferedImage
           [java.io ByteArrayInputStream ByteArrayOutputStream]
           javax.imageio.ImageIO
           org.apache.commons.io.IOUtils))

(def user-agent
  "The user agent string."
  (str "Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 "
       "(KHTML, like Gecko) Chrome/66.0.3359.117 Safari/537.36"))

(def dimensions
  "The dimension used by the photos service to scale images."
  [{:label "small" :width 480}
   {:label "medium" :width 960}
   {:label "large" :width 2048}])

(defprotocol Service
  (-download! [service photo opts])
  (-resize! [service photo dimensions opts]))

(defn download!
  "Create the `photo`."
  [service photo & [opts]]
  (-download! service photo opts))

(defn resize!
  "Resize the `photo`."
  [service photo dimensions & [opts]]
  (-resize! service photo dimensions opts))

(defn- buffered-image->bytes
  "Convert `buffered-image` into a byte in `format`."
  [^BufferedImage buffered-image ^String format]
  (let [baos (ByteArrayOutputStream.)]
    (assert (ImageIO/write buffered-image format baos)
            (str "Can't write image in format: " format))
    (.toByteArray baos)))

(defn- content-md5
  "Return the MD5 sum of `bytes`."
  [bytes]
  (base64-encode (digest/md5-bytes bytes)))

(defn- content-type
  "Return the content type of `response`."
  [response]
  (get-in response [:headers "content-type"]))

(defn content-type-extension
  "Returns the filename extension from `content-type`."
  [content-type]
  (if-not (str/blank? content-type)
    (last (str/split content-type #"/"))))

(defn- content-disposition
  "Return the content disposition header of `photo`."
  [photo label content-type]
  (str "inline; filename=photo-"
       (:id photo) "-" label "."
       (content-type-extension content-type)))

(defn image-storage-key
  "Return the storage key for `photo`, `label` and `content-type`."
  [service photo label content-type]
  (let [extension (content-type-extension content-type)]
    (str "photos" "/" (:id photo) "/" label "." extension)))

(defn image-storage-url
  "Return the storage url for `photo`, `label` and `content-type`."
  [service photo label content-type]
  (format "https://storage.googleapis.com/%s/%s"
          (-> service :storage :bucket)
          (image-storage-key service photo label content-type)))

(defn download-image
  "Download the original image of `photo`."
  [service {:keys [id] :as photo}]
  (let [response (http/get
                  (:url photo)
                  {:as :stream
                   :headers {:user-agent (:user-agent service)}})
        bytes (IOUtils/toByteArray (:body response)) ]
    (log/infof "Downloaded photo %s." id)
    (if-let [image (ImageIO/read (ByteArrayInputStream. bytes))]
      (let [content-type (content-type response)]
        (log/infof "Decoded photo %s as content type %s." id content-type)
        {:buffered image
         :bytes bytes
         :content-type content-type
         :label "original"})
      (throw (ex-info "Can't decode photo."
                      {:type :decode-error
                       :photo photo
                       :response response})))))

(defn save-image-to-db
  "Save the `image` to the database."
  [{:keys [db] :as service} photo image]
  (let [{:keys [buffered content-type label]} image]
    (->> {:photo-id (:id photo)
          :content-disposition (content-disposition photo label content-type)
          :content-length (count (:bytes image))
          :content-md5 (content-md5 (:bytes image))
          :height (.getHeight buffered)
          :storage-key (image-storage-key service photo label content-type)
          :url (image-storage-url service photo label content-type)
          :width (.getWidth buffered)}
         (merge image (images/by-photo-and-label db photo (:label image)))
         (images/save! db)
         (merge image))))

(defn save-image-to-storage
  "Upload the `image`."
  [service photo image]
  (->> {:acl storage/acl-public
        :bytes (:bytes image)
        :cache-control "public, max-age 31536000"
        :content-disposition (:content-disposition image)
        :content-encoding :identity
        :key (:storage-key image)
        :meta-data {:id (str (:id image))
                    :photo-id (str (:id photo))
                    :label (:label image)
                    :width (:width image)
                    :height (:height image)}}
       (storage/save! (:storage service)))
  (log/infof "Saved image #%d to Google Storage at %s."
             (:id image) (:url image))
  image)

(defn resize-image
  "Resize `image` to `dimension`."
  [image dimension]
  (let [{:keys [label width height]} dimension
        buffered (:buffered image)
        resized (cond
                  (and width height)
                  (resizer/resize buffered width height)
                  width
                  (resizer/resize-to-width buffered width)
                  height
                  (resizer/resize-to-height buffered height))]
    {:buffered resized
     :bytes (buffered-image->bytes resized "jpg")
     :content-type "image/jpeg"
     :label label}))

(defn import-image
  "Import the `image` of `photo`."
  [service photo image]
  (->> (save-image-to-db service photo image)
       (save-image-to-storage service photo)))

(defn download-and-import
  "Download the original image of `photo` and import it."
  [service photo]
  (->> (download-image service photo)
       (import-image service photo)))

(defn resize-and-import
  "Resize the `original` image of `photo` to `dimension` and import it."
  [service photo original dimension]
  (->> (resize-image original dimension)
       (import-image service photo)))

(defn update-photo-status
  "Update the `status` of `photo` in the database."
  [service photo status]
  (->> (assoc photo :status (name status))
       (photos/update! (:db service))))

(defn- save-photo! [db photo]
  (if (:id photo)
    (photos/update! db photo)
    (photos/insert! db photo)))

(defrecord Photos [storage user-agent]
  Service
  (-download! [service photo opts]
    (try
      (sql/with-transaction [db (:db service)]
        (let [service (assoc service :db db)
              photo (save-photo! db photo)
              original (download-and-import service photo)]
          (log/infof "Photo %s successfully downloaded." (:id photo))
          (update-photo-status service photo :finished)))
      (catch Throwable e
        (let [msg (format "Photo download error: %s (%s)"
                          (.getMessage e) (:id photo))]
          (log/error msg)
          (throw (ex-info msg {:type ::download-error
                               :photo photo
                               :opts opts}))))))

  (-resize! [service photo dimensions opts]
    (try
      (sql/with-transaction [db (:db service)]
        (let [service (assoc service :db db)
              photo (save-photo! db photo)
              original (download-and-import service photo)]
          (let [images (mapv #(resize-and-import service photo original %) dimensions)]
            (log/infof "Photo %s successfully resized." (:id photo))
            (update-photo-status service photo :finished)
            (assoc photo :images (mapv #(images/by-id db (:id %)) images)))))
      (catch Throwable e
        (let [msg (format "Photo resize error: %s (%s)"
                          (.getMessage e) (:id photo))]
          (log/error msg)
          (throw (ex-info msg {:type ::resize-error
                               :photo photo
                               :opts opts} e)))))))

(defn service
  "Returns a photo service component."
  [& [opts]]
  (map->Photos (merge {:user-agent user-agent} opts)))

(defn create-profile-photo!
  "Create the profile `photo` of `user`."
  [{:keys [db] :as service} user photo & [opts]]
  (sql/with-transaction [db db]
    (let [service (assoc service :db db)
          photo (download! service photo opts)]
      (users/add-profile-photos! db user [photo])
      (log/infof "Profile photo %s for user %s successfully created."
                 (:id photo) (:id user))
      photo)))
