(ns burningswell.streams.weather.gdal
  (:require [burningswell.file :refer [with-tmp-files]]
            [burningswell.shell :refer [sh!]]
            [cheshire.core :as json]
            [clojure.edn :as edn]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [datumbazo.shell :as shell]
            [net.cgrand.enlive-html :as enlive]
            [no.en.core :refer [parse-long]]))

(defn geo-tranform?
  "Return true if `x` is a geo transformation, otherwise false."
  [x]
  (and (= (count x) 6) (every? number? x)))

(defn geo-x
  "Calculate the X geo coordinate."
  [transform raster-x raster-y]
  {:pre [(geo-tranform? transform)]}
  (let [[t0 t1 t2 _ _ _] transform]
    (+ t0 (* raster-x t1) (* raster-y t2))))

(defn geo-y
  "Calculate the Y geo coordinate."
  [transform raster-x raster-y]
  {:pre [(geo-tranform? transform)]}
  (let [[_ _ _ t3 t4 t5] transform]
    (+ t3 (* raster-x t4) (* raster-y t5))))

(defn raster-x
  "Calculate the X raster coordinate."
  [transform geo-x geo-y]
  {:pre [(geo-tranform? transform)]}
  (let [[t0 t1 t2 _ _ _] transform]
    (/ (- geo-x (* geo-y t2) t0) t1)))

(defn raster-y
  "Calculate the Y raster coordinate."
  [transform geo-x geo-y]
  {:pre [(geo-tranform? transform)]}
  (let [[_ _ _ t3 t4 t5] transform]
    (/ (- geo-y t3 (* geo-x t4)) t5)))

(defn geo-lower-left
  "Return the geo coordinates at the lower left of `rect`."
  [transform rect]
  {:pre [(geo-tranform? transform)]}
  [(geo-x transform (:xOff rect) (:ySize rect))
   (geo-y transform (:xOff rect) (:ySize rect))])

(defn geo-lower-right
  "Return the geo coordinates at the lower right of `rect`."
  [transform rect]
  {:pre [(geo-tranform? transform)]}
  [(geo-x transform (:xSize rect) (:ySize rect))
   (geo-y transform (:xSize rect) (:ySize rect))])

(defn geo-upper-left
  "Return the geo coordinates at the upper left of `rect`."
  [transform rect]
  {:pre [(geo-tranform? transform)]}
  [(geo-x transform (:xOff rect) (:yOff rect))
   (geo-y transform (:xOff rect) (:yOff rect))])

(defn geo-upper-right
  "Return the geo coordinates at the upper right of `rect`."
  [transform rect]
  {:pre [(geo-tranform? transform)]}
  [(geo-x transform (:xSize rect) (:yOff rect))
   (geo-y transform (:xSize rect) (:yOff rect))])

(defn geo-center
  "Return the geo coordinates at the center of `rect`."
  [transform rect]
  {:pre [(geo-tranform? transform)]}
  (let [x-center (/ (:xSize rect) 2.0)
        y-center (/ (:ySize rect) 2.0)]
    [(geo-x transform x-center y-center)
     (geo-y transform x-center y-center)]))

(defn geo-corner-coordinates
  "Return the geo corner coordinates at the center of `rect`."
  [transform rect]
  {:pre [(geo-tranform? transform)]}
  {:center (geo-center transform rect)
   :lower-left (geo-lower-left transform rect)
   :lower-right (geo-lower-right transform rect)
   :upper-left (geo-upper-left transform rect)
   :upper-right (geo-upper-right transform rect)})

(defn parse-geo-transform
  "Parse the geo transformation from `node`."
  [node]
  {:post [(geo-tranform? %)]}
  (when-let [content (first (:content node))]
    (->> (str/split content #"\s*,\s*")
         (mapv edn/read-string))))

(defn select-geo-transform
  "Return the geo transformation from `document`."
  [document]
  {:post [(geo-tranform? %)]}
  (when-let [node (first (enlive/select document [:GeoTransform]))]
    (parse-geo-transform node)))

(defn dataset-rect
  "Return the rectangle of the dataset from `document`."
  [document]
  (when-let [attrs (-> (enlive/select document [:VRTDataset]) first :attrs)]
    {:xOff 0
     :yOff 0
     :xSize (-> attrs :rasterXSize parse-long)
     :ySize (-> attrs :rasterYSize parse-long)}))

(defn update-geo-transform
  "Select the GeoTransform node in `document`."
  [rectangle]
  (fn [node]
    (let [[t0 t1 t2 t3 t4 t5 :as transform] (parse-geo-transform node)
          [ul-x ul-y] (geo-upper-left transform rectangle)
          [ur-x ur-y] (geo-upper-right transform rectangle)
          new-t0 (+ (/ (- ul-x ur-x ) 2) t0)
          new-transform [new-t0 t1 t2 t3 t4 t5]]
      (assoc node :content [(str/join ", " new-transform)]))))

(defn update-src-rect
  [transform]
  (fn [node]
    node))

(defn update-dest-rect
  [transform]
  (fn [node]
    node))

(defn- select-attributes
  "Select the attributes of `tag` in `node`."
  [node tag]
  (-> (enlive/select node [tag]) first :attrs))

(defn source-properties
  "Return the source properties from `node`."
  [node]
  (reduce #(update %1 %2 parse-long)
          (select-attributes node :SourceProperties)
          [:BlockXSize :BlockYSize :RasterXSize :RasterYSize]))

(defn rectangle-attributes
  "Return the rectangle attributes of `tag` from `node`."
  [node tag]
  (reduce #(update %1 %2 parse-long)
          (select-attributes node tag)
          [:xOff :yOff :xSize :ySize]))

(defn crosses-dateline?
  "Returns true if `rect` crosses the dateline under `transform`,
  otherwise false."
  [transform rect]
  (> (first (geo-upper-right transform rect)) 180))

(defn- set-attributes
  "Set the attributes of a node to `attrs`."
  [attrs]
  #(assoc % :attrs attrs))

(defn eastern-source
  "Return the new eastern source rectangle."
  [rect]
  (let [half-x (int (/ (:xSize rect) 2))]
    {:xOff half-x
     :xSize half-x
     :yOff (:yOff rect)
     :ySize (:ySize rect)}))

(defn eastern-destination
  "Return the new eastern destination rectangle."
  [rect]
  (let [half-x (int (/ (:xSize rect) 2))]
    {:xOff (:xOff rect)
     :xSize half-x
     :yOff (:yOff rect)
     :ySize (:ySize rect)}))

(defn western-source
  "Return the new western source rectangle."
  [rect]
  (let [half-x (int (/ (:xSize rect) 2))]
    {:xOff (:xOff rect)
     :xSize half-x
     :yOff (:yOff rect)
     :ySize (:ySize rect)}))

(defn western-destination
  "Return the new western destination rectangle."
  [rect]
  (let [half-x (int (/ (:xSize rect) 2))]
    {:xOff half-x
     :xSize half-x
     :yOff (:yOff rect)
     :ySize (:ySize rect)}))

(defn update-rectangle [node tag f]
  (let [rect (rectangle-attributes node tag)]
    (enlive/at node [tag] (set-attributes (f rect)))))

(defn simple-eastern-source
  "Make a new simple source for the eastern part of the raster."
  [node]
  (-> (update-rectangle node :SrcRect eastern-source)
      (update-rectangle :DstRect eastern-destination)
      (first)))

(defn simple-western-source
  "Make a new simple source for the western part of the raster."
  [node]
  (-> (update-rectangle node :SrcRect western-source)
      (update-rectangle :DstRect western-destination)
      (first)))

(defn update-simple-source
  "Update the simple source of a raster band."
  [transform]
  (fn [node]
    (let [src-rect (rectangle-attributes node :SrcRect)
          dst-rect (rectangle-attributes node :DstRect)]
      (assert (= src-rect dst-rect))
      (if (crosses-dateline? transform src-rect)
        [(simple-eastern-source node)
         "\n    "
         (simple-western-source node)]
        node))))

(defn update-raster-band
  "Update the raster band."
  [transform]
  #(enlive/at % [:SimpleSource] (update-simple-source transform)))

(defn render-xml
  "Render `document` as a XML string."
  [document]
  (apply str (enlive/emit* document)))

(defn center-virtual-format-xml
  "Center the rasters in the GDAL virtual format `input` and return
  the XML as a string."
  [input]
  (let [document (enlive/xml-resource input)
        transform (select-geo-transform document)
        rectangle (dataset-rect document)]
    (if (crosses-dateline? transform rectangle)
      (-> (enlive/at document [:GeoTransform] (update-geo-transform rectangle))
          (enlive/at [:VRTRasterBand] (update-raster-band transform))
          (render-xml))
      (render-xml document))))

(defn center-virtual-format-file
  "Read the GDAL virtual format file from `input`, shift the content
  into the -180,180 longitude range and write a new virtual format
  file to `output`."
  [input output]
  (with-tmp-files [tmp-file]
    (io/make-parents output)
    (->> (center-virtual-format-xml (io/file input))
         (spit tmp-file))
    (io/copy (io/file tmp-file) (io/file output))))

(defn create-virtual-format
  "Create a GDAL virtual format file for `input` with the
  gdal_translate command and write it to `output`."
  [input output & [{:keys [center? srid]}]]
  (let [srid (str "EPSG:" (or srid 4326))]
    (with-tmp-files [vrt-file]
      (io/make-parents output)
      (shell/exec-checked-script
       "Create GDAL VRT file"
       ("gdal_translate"
        "-of" "VRT"
        "-a_srs" ~(str srid)
        ~(str input) ~(str vrt-file)))
      (if center?
        (center-virtual-format-file vrt-file output)
        (io/copy (io/file vrt-file) (io/file output))))))

(defn to-mercator
  "Project `input` into Mercator and write the file to `output`."
  [input output]
  (.delete (io/file output))
  (io/make-parents output)
  (shell/exec-checked-script
   "Transform to Mercator."
   ("gdalwarp"
    "-t_srs" "EPSG:3395"
    "-r" "lanczos"
    "-wo" "SOURCE_EXTRA=1000"
    ;; "-co" "COMPRESS=LZW"
    "-of" "VRT"
    ~(str input) ~(str output))))

(defn info
  "Return the GDAL information about `file` as a map."
  [file]
  (-> (sh! "gdalinfo" "-json" "-mm" "-proj4" (str file))
      :out (json/parse-string keyword)))
