(ns burningswell.streams.page-views
  (:require [burningswell.api.specs.events]
            [burningswell.streams.core :as k]
            [peripheral.core :refer [defcomponent]])
  (:import [org.apache.kafka.streams KafkaStreams StreamsConfig]))

(defn config [& [opts]]
  (->> {:application.id "url-views"
        :input {:events "burningswell.api.events"}
        :output {:total "burningswell.page-views.urls.total"
                 :total-per-user "burningswell.page-views.urls.total-per-user"}}
       (merge opts)))

(defn- start-stream [{:keys [config topology] :as env}]
  (doto (KafkaStreams. topology (StreamsConfig. (k/props config)))
    (.start)))

(defn- total-key [key value]
  {:url (str (:url value))})

(defn- total-per-user-key [key value]
  {:url (str (:url value))
   :user-id (:user-id value)})

(defn- authenticated? [key value]
  (some? (:user-id value)))

(defn- page-view-event? [key event]
  (= (:name event) :burningswell.api.events/page-view-created))

(defn- page-view-events [env builder]
  (-> (.stream builder (-> env :config :input :events))
      (k/filter page-view-event?)))

(defn total [env input]
  (-> (k/group-by input total-key)
      (.count (-> env :config :output :total))))

(defn total-per-user [env input]
  (-> (k/filter input authenticated?)
      (k/group-by total-per-user-key)
      (.count (-> env :config :output :total-per-user))))

(defn make-topology [env]
  (k/with-build-stream builder
    (let [page-views (page-view-events env builder)]
      (total env page-views)
      (total-per-user env page-views))))

(defcomponent PageViews [config]
  :this/as *this*
  :topology (make-topology *this*)
  :stream (start-stream *this*) #(.close %))

(defn page-views [& [opts]]
  (map->PageViews {:config (config opts)}))
