(ns burningswell.services.storage
  (:refer-clojure :exclude [get])
  (:require [clojure.spec.alpha :as s]
            [clojure.tools.logging :as log]
            [clojure.walk :as walk]
            [pandect.core :as digest]
            [ring.util.codec :refer [base64-encode]])
  (:import [com.google.cloud.storage Acl Acl$Role Acl$User
            Blob$BlobSourceOption BlobId BlobInfo
            Storage$BlobTargetOption StorageOptions]))

(defprotocol Storage
  (-get [storage blob])
  (-delete! [storage blob])
  (-save! [storage blob]))

(s/def ::acl (s/nilable vector?))
(s/def ::bucket string?)
(s/def ::cache-control (s/nilable string?))
(s/def ::content-encoding (s/nilable keyword?))
(s/def ::key string?)
(s/def ::md5 (s/nilable string?))
(s/def ::meta-data (s/nilable map?))

(s/def ::blob
  (s/keys :req-un [::key]
          :opt-un [::acl
                   ::cache-control
                   ::content-encoding
                   ::md5
                   ::meta-data]))

(s/def ::storage #(satisfies? Storage %))

(defn get
  "Get the `blob` from object storage."
  [storage blob]
  (-get storage blob))

(s/fdef get
  :args (s/cat :storage ::storage :blob ::blob)
  :ret ::blob)

(defn delete!
  "Delete the `blob` from object storage."
  [storage blob]
  (let [result (-delete! storage blob)]
    (log/infof "Deleted object from storage: %s" (:url result))
    result))

(s/fdef delete!
  :args (s/cat :storage ::storage :blob ::blob)
  :ret ::blob)

(defn save!
  "Save the `blob` to object storage."
  [storage blob]
  (let [result (-save! storage blob)]
    (log/infof "Saved object to storage: %s" (:url result))
    result))

(s/fdef save!
  :args (s/cat :storage ::storage :blob ::blob)
  :ret ::blob)

(def acl-public
  [(Acl/of (Acl$User/ofAllUsers) Acl$Role/READER)])

(defn acl
  "Returns the access control list for `blob`."
  [blob]
  (or (:acl blob) acl-public))

(defn cache-control
  "Returns the cache control for `blob`."
  [blob]
  (or (:cache-control blob) "public, max-age 31536000"))

(defn content-encoding
  "Returns the content encoding for `blob`."
  [blob]
  (name (or (:content-encoding blob) :identity)))

(defn md5
  "Return the MD5 sum for `blob`."
  [blob]
  (some-> blob :bytes digest/md5-bytes base64-encode))

(defn meta-data
  "Return the MD5 sum for `blob`."
  [blob]
  (walk/stringify-keys (:meta-data blob)))

;; Google Cloud Storage

(defn- blob-info
  "Returns the `blob` info as a map."
  [blob]
  {:acl (acl blob)
   :cache-control (cache-control blob)
   :content-disposition (:content-disposition blob)
   :content-encoding (content-encoding blob)
   :content-type (:content-type blob)
   :key (:key blob)
   :md5 (md5 blob)
   :meta-data (meta-data blob)})

(defn- blob->map
  "Returns the `blob` as a map."
  [blob]
  (when blob
    {:key (.getName blob)
     :bytes (.getContent blob (into-array Blob$BlobSourceOption []))}))

(defn- make-blob-info
  "Returns the `blob` info for `storage`."
  [{:keys [bucket] :as storage} blob]
  (let [info (blob-info blob)]
    (-> (BlobInfo/newBuilder bucket (:key info))
        (.setAcl (:acl info))
        (.setCacheControl (:cache-control info))
        (.setContentDisposition (:content-disposition info))
        (.setContentEncoding (:content-encoding info))
        (.setContentType (:content-type info))
        (.setMd5 (:md5 info))
        (.setMetadata (:meta-data info))
        (.build))))

(defn- google-storage-url
  "Return the `storage` url for `blob`."
  [storage blob]
  (format "https://storage.googleapis.com/%s/%s" (:bucket storage) (:key blob)))

(defn- service [storage]
  (.. StorageOptions getDefaultInstance getService))

(defn- blob-id [storage blob]
  (BlobId/of (:bucket storage) (:key blob)))

(defrecord Google [bucket]
  Storage
  (-delete! [storage blob]
    (.delete (service storage) (blob-id storage blob))
    (assoc blob :url (google-storage-url storage blob)))

  (-get [storage blob]
    (blob->map (.get (service storage) (blob-id storage blob))))

  (-save! [storage blob]
    (.create (service storage)
             (make-blob-info storage blob)
             (:bytes blob)
             (into-array Storage$BlobTargetOption []))
    (assoc blob :url (google-storage-url storage blob))))

(defn google
  "Returns a new Google Cloud storage."
  [config]
  (map->Google config))

(s/fdef google
  :args (s/cat :config (s/keys :req-un [::bucket]))
  :ret ::storage)

;; In memory storage

(defrecord InMemory [bucket state]
  Storage
  (-get [storage blob]
    (get-in @state [bucket (:key blob)]))

  (-delete! [storage blob]
    (let [blob (assoc blob :url (google-storage-url storage blob))]
      (swap! state update bucket dissoc (:key blob))
      blob))

  (-save! [storage blob]
    (let [blob (assoc blob :url (google-storage-url storage blob))]
      (swap! state assoc-in [bucket (:key blob)] blob)
      blob)))

(defn in-memory
  "Returns a new in-memory storage."
  [config]
  (map->InMemory (merge {:state (atom {})} config)))

(s/fdef in-memory
  :args (s/cat :config (s/keys :req-un [::bucket]))
  :ret ::storage)
