(ns burningswell.scheduler.system
  (:require [burningswell.cli.backup :as backup]
            [burningswell.cli.natural-earth :as natural-earth]
            [burningswell.cli.weather.import :as weather-import]
            [burningswell.config.core :as config :refer [config]]
            [burningswell.db.connection :refer [with-db]]
            [burningswell.db.countries :as countries]
            [burningswell.db.materialized-views :as materialized-views]
            [burningswell.db.regions :as regions]
            [burningswell.scheduler.quartz :as quartz]
            [burningswell.system :refer [defsystem]]
            [clojure.tools.logging :as log]
            [clojurewerkz.quartzite.jobs :as j]
            [clojurewerkz.quartzite.jobs :refer [defjob]]
            [clojurewerkz.quartzite.schedule.daily-interval :as daily]
            [clojurewerkz.quartzite.schedule.simple :as s]
            [clojurewerkz.quartzite.scheduler :as scheduler]
            [clojurewerkz.quartzite.triggers :as t]
            [environ.core :refer [env]]))

(defjob backup
  [context]
  (backup/-main "create"))

(defjob country-photos
  [context]
  (with-db [db (config/db env)]
    @(countries/update-photos db)
    (log/info "Country photos successfully updated.")))

(defjob materialized-views
  [context]
  (materialized-views/-main
   "--concurrently" "search.autocomplete"))

(defjob natural-earth
  [context]
  (natural-earth/-main))

(defjob region-photos
  [context]
  (with-db [db (config/db env)]
    @(regions/update-photos db)
    (log/info "Region photos successfully updated.")))

(defjob weather
  [context]
  (weather-import/-main "--models" "nww3,gfs"))

(defn schedule-backup
  "Schedule the backup job."
  [quartz]
  (scheduler/schedule
   (:scheduler quartz)
   (j/build
    (j/of-type backup)
    (j/with-identity (j/key "db.backup")))
   (t/build
    (t/with-identity (t/key "db.backup"))
    (t/with-schedule
      (s/schedule
       (s/with-interval-in-days 1)
       (s/repeat-forever)))))
  quartz)

(defn schedule-country-photos
  "Schedule the update of country photos."
  [quartz]
  (scheduler/schedule
   (:scheduler quartz)
   (j/build
    (j/of-type country-photos)
    (j/with-identity (j/key "db.photos.countries")))
   (t/build
    (t/with-identity (t/key "db.photos.countries"))
    (t/with-schedule
      (s/schedule
       (s/with-interval-in-minutes 5)
       (s/repeat-forever)))))
  quartz)

(defn schedule-materialized-views
  "Schedule the materialized views job."
  [quartz]
  (scheduler/schedule
   (:scheduler quartz)
   (j/build
    (j/of-type materialized-views)
    (j/with-identity (j/key "db.materialized-views")))
   (t/build
    (t/with-identity (t/key "db.materialized-views"))
    (t/with-schedule
      (s/schedule
       (s/with-interval-in-minutes 5)
       (s/repeat-forever)))))
  quartz)

(defn schedule-natural-earth
  "Schedule the natural earth job."
  [quartz]
  (scheduler/schedule
   (:scheduler quartz)
   (j/build
    (j/of-type natural-earth)
    (j/with-identity (j/key "db.natural-earth")))
   (t/build
    (t/with-identity (t/key "db.natural-earth"))
    (t/with-schedule
      (s/schedule
       (s/with-interval-in-days 1)
       (s/repeat-forever)))))
  quartz)

(defn schedule-region-photos
  "Schedule the update of region photos."
  [quartz]
  (scheduler/schedule
   (:scheduler quartz)
   (j/build
    (j/of-type region-photos)
    (j/with-identity (j/key "db.photos.regions")))
   (t/build
    (t/with-identity (t/key "db.photos.regions"))
    (t/with-schedule
      (s/schedule
       (s/with-interval-in-minutes 5)
       (s/repeat-forever)))))
  quartz)

(defn schedule-weather
  "Schedule the weather job."
  [quartz]
  (scheduler/schedule
   (:scheduler quartz)
   (j/build
    (j/of-type weather)
    (j/with-identity (j/key "weather.datasets")))
   (t/build
    (t/with-identity (t/key "weather.datasets"))
    (t/with-schedule
      (s/schedule
       (s/with-interval-in-hours 4)
       (s/repeat-forever)))))
  quartz)

(defn register
  "Register all jobs with the scheduler."
  [quartz]
  (-> quartz
      (schedule-backup)
      (schedule-country-photos)
      (schedule-region-photos)
      (schedule-materialized-views)
      (schedule-natural-earth)
      (schedule-weather)))

(defsystem system
  "Make a new scheduler system."
  []
  (quartz/quartz {:schedule-fn register}))
