(ns burningswell.scheduler.system
  (:require [burningswell.config.core :refer [config]]
            [burningswell.db.backup :as backup]
            [burningswell.db.natural-earth :as natural-earth]
            [burningswell.db.materialized-views :as materialized-views]
            [burningswell.scheduler.quartz :as quartz]
            [burningswell.weather.import :as weather]
            [burningswell.system :refer [defsystem]]
            [clojurewerkz.quartzite.jobs :as j]
            [clojurewerkz.quartzite.jobs :refer [defjob]]
            [clojurewerkz.quartzite.schedule.daily-interval :as daily]
            [clojurewerkz.quartzite.scheduler :as scheduler]
            [clojurewerkz.quartzite.schedule.simple :as s]
            [clojurewerkz.quartzite.triggers :as t]
            [environ.core :refer [env]]))

(defjob backup
  [context]
  (backup/-main "create"))

(defjob materialized-views
  [context]
  (materialized-views/-main
   "--concurrently" "search.autocomplete"))

(defjob natural-earth
  [context]
  (natural-earth/-main))

(defjob weather
  [context]
  (weather/-main "--models" "nww3,gfs"))

(defn schedule-backup
  "Schedule the backup job."
  [quartz]
  (scheduler/schedule
   (:scheduler quartz)
   (j/build
    (j/of-type backup)
    (j/with-identity (j/key "db.backup")))
   (t/build
    (t/with-identity (t/key "db.backup"))
    (t/with-schedule
      (s/schedule
       (s/with-interval-in-days 1)
       (s/repeat-forever)))))
  quartz)

(defn schedule-materialized-views
  "Schedule the materialized views job."
  [quartz]
  (scheduler/schedule
   (:scheduler quartz)
   (j/build
    (j/of-type materialized-views)
    (j/with-identity (j/key "db.materialized-views")))
   (t/build
    (t/with-identity (t/key "db.materialized-views"))
    (t/with-schedule
      (s/schedule
       (s/with-interval-in-minutes 1)
       (s/repeat-forever)))))
  quartz)

(defn schedule-natural-earth
  "Schedule the natural earth job."
  [quartz]
  (scheduler/schedule
   (:scheduler quartz)
   (j/build
    (j/of-type natural-earth)
    (j/with-identity (j/key "db.natural-earth")))
   (t/build
    (t/with-identity (t/key "db.natural-earth"))
    (t/with-schedule
      (s/schedule
       (s/with-interval-in-days 1)
       (s/repeat-forever)))))
  quartz)

(defn schedule-weather
  "Schedule the weather job."
  [quartz]
  (scheduler/schedule
   (:scheduler quartz)
   (j/build
    (j/of-type weather)
    (j/with-identity (j/key "weather.datasets")))
   (t/build
    (t/with-identity (t/key "weather.datasets"))
    (t/with-schedule
      (s/schedule
       (s/with-interval-in-days 1)
       (s/repeat-forever)))))
  quartz)

(defn register
  "Register all jobs with the scheduler."
  [quartz]
  (-> (schedule-backup quartz)
      (schedule-materialized-views)
      (schedule-natural-earth)
      (schedule-weather)))

(defsystem system
  "Make a new scheduler system."
  []
  (quartz/quartz {:schedule-fn register}))
