(ns burningswell.rabbitmq.client
  (:require [clojure.edn :as edn]
            [clojure.set :as set]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [kithara.rabbitmq.channel :as channel]
            [kithara.rabbitmq.connection :as connection]
            [kithara.rabbitmq.publish :as publish]
            [no.en.core :as noencore])
  (:import com.rabbitmq.client.Connection))

(def ^:dynamic *defaults*
  "The default RabbitMQ client configuration."
  {:data-readers *data-readers*
   :scheme :amqp
   :server-name "localhost"
   :server-port 5672
   :username "guest"
   :password "guest"
   :vhost "/"})

(defrecord Client
    [data-readers
     password
     scheme
     server-name
     server-port
     username
     vhost])

(defn client?
  "Returns true if `x` is a RabbitMQ client, otherwise false."
  [x]
  (instance? Client x))

(defn read-edn
  ([payload]
   (read-edn payload {:readers *data-readers*}))
  ([payload opts]
   (edn/read-string opts (String. payload "UTF-8"))))

(defn decode
  "Decode the EDN `payload` using the :data-readers of `client`."
  [client payload]
  (read-edn payload {:readers (:data-readers client)}))

(defn config
  "Return the Kithara config for `client`."
  [config]
  (set/rename-keys config {:server-name :host :server-port :port}))

(defn public-url
  "Return the formatted public url of `client`."
  [client]
  (noencore/public-url (set/rename-keys client {:vhost :uri})))

(defn connected?
  "Return true if `client` is connected, false otherwise."
  [client]
  (instance? Connection (:connection client)))

(defn start-client
  "Start the RabbitMQ client."
  [client]
  (if (connected? client)
    client
    (let [connection (connection/open (config client))]
      (log/debugf "RabbitMQ client to %s started." (public-url client))
      (assoc client :connection connection))))

(defn stop-client
  "Stop the RabbitMQ client."
  [client]
  (when-let [connection (:connection client)]
    (when (.isOpen connection)
      (connection/close connection))
    (log/debugf "RabbitMQ client to %s stopped." (public-url client)))
  (dissoc client :connection))

(defn client
  "Return a RabbitMQ client."
  [& [config]]
  (map->Client (merge *defaults* config)))

(defmacro with-connection
  "Start the `component`, bind the started component to
  `component-sym`, evaluate `body` and stop `component` again."
  [[client-sym config] & body]
  `(let [component# (component/start (client ~config))
         ~client-sym component#]
     (try ~@body
          (finally (component/stop component#)))))

(defmacro with-channel
  "Start the `component`, bind the started component to
  `component-sym`, evaluate `body` and stop `component` again."
  [[channel-sym client] & body]
  `(let [channel# (channel/open (:connection ~client))
         ~channel-sym channel#]
     (try ~@body
          (finally (channel/close channel#)))))

(defn publish-edn
  "Return true if `client` is connected, false otherwise."
  [broker-or-channel message]
  (if (client? broker-or-channel)
    (with-channel [channel broker-or-channel]
      (publish-edn channel message))
    (->> (update-in message [:body] #(some-> % pr-str .getBytes))
         (publish/publish broker-or-channel))))

(defn purge-queue
  "Purge a queue."
  [channel queue]
  (.getMessageCount (.queuePurge channel  (name queue))))


(extend-type Client
  component/Lifecycle
  (start [client]
    (start-client client))
  (stop [client]
    (stop-client client)))
