(ns burningswell.rabbitmq.client
  (:require [burningswell.component :refer [with-component]]
            [clojure.edn :as edn]
            [clojure.set :as set]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [kithara.rabbitmq.channel :as channel]
            [kithara.rabbitmq.exchange :as exchange]
            [kithara.rabbitmq.connection :as connection]
            [kithara.rabbitmq.publish :as publish]
            [kithara.rabbitmq.queue :as queue]
            [no.en.core :as noencore]
            [schema.core :as s])
  (:import [com.rabbitmq.client Channel Connection]))

(def ^:dynamic *defaults*
  "The default RabbitMQ client configuration."
  {:data-readers *data-readers*
   :scheme :amqp
   :server-name "localhost"
   :server-port 5672
   :username "guest"
   :password "guest"
   :vhost "/"})

(s/defrecord Client
    [channel :- (s/maybe Channel)
     connection :- (s/maybe Connection)
     data-readers :- s/Any
     password :- s/Str
     scheme :- s/Keyword
     server-name :- s/Str
     server-port :- s/Int
     username :- s/Str
     vhost :- s/Str]
  {s/Any s/Any})

(defn read-edn
  ([payload]
   (read-edn payload {:readers *data-readers*}))
  ([payload opts]
   (edn/read-string opts (String. payload "UTF-8"))))

(defn decode
  "Decode the EDN `payload` using the :data-readers of `client`."
  [client payload]
  (read-edn payload {:readers (:data-readers client)}))

(defn config
  "Return the Kithara config for `client`."
  [config]
  (set/rename-keys config {:server-name :host :server-port :port}))

(s/defn ^:private public-url :- s/Str
  "Return the formatted public url of `client`."
  [client :- Client]
  (-> (assoc client :uri (str "/" (:vhost client)))
      (noencore/public-url)))

(s/defn start-client :- Client
  "Start the RabbitMQ client."
  [client :- Client]
  (if (:channel client)
    client
    (let [connection (connection/open (config client))
          channel (channel/open connection)]
      (log/infof "RabbitMQ client to %s started." (public-url client))
      (assoc client :connection connection :channel channel))))

(s/defn stop-client :- Client
  "Stop the RabbitMQ client."
  [client :- Client]
  (when-let [channel (:channel client)]
    (when (.isOpen channel)
      (channel/close channel)))
  (when-let [connection (:connection client)]
    (when (.isOpen connection)
      (connection/close connection))
    (log/infof "RabbitMQ client to %s stopped." (public-url client)))
  (assoc client :connection nil :channel nil))

(s/defn connected? :- s/Bool
  "Return true if `client` is connected, false otherwise."
  [client :- Client]
  (boolean (and (:connection client) (:channel client))))

(s/defn client :- Client
  "Return a RabbitMQ client."
  [& [config]]
  (map->Client (merge *defaults* config)))

(defmacro with-client
  "Start the RabbitMQ `client`, bind the started instance to
  `client-sym`, evaluate `body` and stop the client again."
  [[client-sym config] & body]
  `(with-component [~client-sym (client ~config)]
     ~@body))

(s/defn bind
  "Bind a queue."
  [queue & [opts]]
  (queue/bind queue opts))

(s/defn delete-exchange
  "Delete an exchange."
  ([client :- Client exchange :- s/Str & [opts]]
   {:pre [(connected? client)]}
   (exchange/delete
    {:channel (:channel client)
     :exchange-name exchange})))

(s/defn declare-exchange
  "Declare an exchange."
  ([client :- Client exchange :- s/Str type :- s/Keyword & [opts]]
   {:pre [(connected? client)]}
   (exchange/declare (:channel client) exchange type opts)))

(s/defn declare-queue
  "Declare a queue."
  ([client :- Client]
   {:pre [(connected? client)]}
   (queue/declare (:channel client)))
  ([client :- Client queue :- s/Str & [opts]]
   {:pre [(connected? client)]}
   (queue/declare (:channel client) queue opts)))

(s/defn purge-queue
  "Purge a queue."
  [client :- Client queue :- s/Str]
  {:pre [(connected? client)]}
  (.getMessageCount (.queuePurge (:channel client) queue)))

(s/defn publish-message
  "Publish a message."
  [client :- Client message]
  {:pre [(connected? client)]}
  (->> (update-in message [:body] #(some-> % pr-str .getBytes))
       (publish/publish (:channel client))))

(extend-type Client
  component/Lifecycle
  (start [client]
    (start-client client))
  (stop [client]
    (stop-client client)))
