(ns burningswell.rabbitmq.core
  (:require [clojure.edn :as edn]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [langohr.basic :as basic]
            [langohr.channel :as channel]
            [langohr.consumers :as consumers]
            [langohr.core :as core]
            [langohr.exchange :as exchange]
            [langohr.queue :as queue]
            [schema.core :as s])
  (:import [com.novemberain.langohr Connection]
           [com.rabbitmq.client Channel]
           [java.util.concurrent Executors]))

(def ^:dynamic *defaults*
  "The default RabbitMQ config."
  {:password "guest"
   :server-name "localhost"
   :server-port 5672
   :username "guest"
   :vhost "/"
   :threads 1})

(s/defrecord RabbitMQ
    [connection :- (s/maybe Connection)
     username :- s/Str
     password :- s/Str
     server-name :- s/Str
     server-port :- s/Int
     vhost :- s/Str
     threads :- s/Int]
  {s/Any s/Any})

(defmulti decode
  (fn [payload metadata]
    (:content-type metadata)))

(defmethod decode "application/edn" [payload metadata]
  (edn/read-string {:readers *data-readers*} (String. payload "UTF-8")))

(defmethod decode :default [payload metadata]
  payload)

(s/defn ^:always-validate bind-queue
  "Bind a RabbitMQ queue to an exchange."
  [channel :- Channel queue :- s/Str exchange :- s/Str & [opts]]
  (queue/bind channel queue exchange opts))

(s/defn ^:always-validate connect :- Connection
  "Connect to RabbitMQ."
  [rabbitmq :- RabbitMQ]
  (core/connect
   {:executor (Executors/newFixedThreadPool (:threads rabbitmq))
    :host (:server-name rabbitmq)
    :password (:password rabbitmq)
    :port (:server-port rabbitmq)
    :username (:username rabbitmq)
    :vhost (:vhost rabbitmq)}))

(s/defn ^:always-validate publish
  "Publish a message to RabbitMQ."
  [channel :- Channel
   exchange :- s/Str
   routing-key :- s/Str
   payload :- s/Any & [opts]]
  (let [opts (assoc opts :content-type "application/edn")]
    (basic/publish channel exchange routing-key (pr-str payload) opts)))

(s/defn ^:always-validate subscribe :- s/Str
  "Subscribe to a RabbitMQ queue."
  [channel :- Channel queue :- s/Str f :- s/Any & [opts]]
  (consumers/subscribe
   channel (name queue)
   (fn [channel metadata payload]
     (f channel metadata (decode payload metadata)))
   opts))

(s/defn ^:always-validate unsubscribe
  "Unsubscribe from a RabbitMQ queue."
  [channel :- Channel consumer-tag :- s/Str]
  (basic/cancel channel consumer-tag))

(s/defn ^:always-validate open-channel :- Channel
  "Open a RabbitMQ channel."
  [rabbitmq :- RabbitMQ]
  (channel/open (:connection rabbitmq)))

(s/defn ^:always-validate close-channel
  "Close the RabbitMQ `channel`."
  [channel :- Channel]
  (channel/close channel))

(s/defn ^:always-validate declare-queue :- s/Str
  "Declare a RabbitMQ queue."
  [channel :- Channel & [name :- s/Str opts]]
  (:queue (if name
            (queue/declare channel name opts)
            (queue/declare channel))))

(s/defn ^:always-validate declare-exchange :- {}
  "Declare a RabbitMQ exchange."
  [channel :- Channel  name :- s/Str type :- s/Str & [opts]]
  (exchange/declare channel name type opts))

(s/defn ^:always-validate start-rabbitmq :- RabbitMQ
  "Start the RabbitMQ component."
  [rabbitmq :- RabbitMQ]
  (if (:connection rabbitmq)
    rabbitmq
    (let [connection (connect rabbitmq)]
      (log/infof
       "RabbitMQ connection to amqp://%s:%s%s established."
       (:server-name rabbitmq)
       (:server-port rabbitmq)
       (:vhost rabbitmq))
      (assoc rabbitmq :connection connection))))

(s/defn ^:always-validate stop-rabbitmq :- RabbitMQ
  "Stop the RabbitMQ component."
  [rabbitmq :- RabbitMQ]
  (when-let [connection (:connection rabbitmq)]
    (log/infof
     "RabbitMQ connection to amqp://%s:%s%s closed."
     (:server-name rabbitmq)
     (:server-port rabbitmq)
     (:vhost rabbitmq))
    (.close connection))
  (assoc rabbitmq :connection nil))

(extend-protocol component/Lifecycle
  RabbitMQ
  (start [rabbitmq]
    (start-rabbitmq rabbitmq))
  (stop [rabbitmq]
    (stop-rabbitmq rabbitmq)))

(s/defn ^:always-validate rabbitmq :- RabbitMQ
  "Make a new RabbitMQ component."
  [& [config]]
  (map->RabbitMQ (merge *defaults* config)))

(defmacro with-rabbitmq
  "Create a new RabbitMQ component, bind the started component to
  `rabbitmq-sym`, evaluate `body` and stop the component again."
  [[rabbitmq-sym config] & body]
  `(let [rabbitmq# (component/start (rabbitmq ~config))
         ~rabbitmq-sym rabbitmq#]
     (try ~@body
          (finally (component/stop rabbitmq#)))))

(defmacro with-channel
  "Open a RabbitMQ channel, bind it to `channel-sym`, evaluate `body`
  and close the channel again."
  [[channel-sym rabbitmq] & body]
  `(let [~channel-sym (open-channel ~rabbitmq)]
     (try ~@body
          (finally (close-channel ~channel-sym)))))

(s/defn ^:always-validate wait-for-empty-queue
  "Wait until queue is empty."
  [channel :- Channel queue :- s/Str timeout-ms :- s/Int]
  (let [started-at (System/currentTimeMillis)]
    (loop [count (queue/message-count channel queue)]
      (if (and (pos? count)
               (< (System/currentTimeMillis)
                  (+ started-at timeout-ms)))
        (recur (queue/message-count channel queue))
        count))))
