(ns burningswell.pedestal.server
  (:require [com.stuartsierra.component :as component]
            [io.pedestal.http :as http]))

(defn init
  "Initialize `server` before starting."
  [server]
  (if (ifn? (:init server))
    ((:init server) server) server))

(defn start-server
  "Start the Pedestal server. "
  [server]
  (if (::started? server)
    server
    (-> (init server)
        (http/create-server)
        (http/start)
        (assoc ::started? true))))

(defn stop-server
  "Stop the Pedestal server. "
  [server]
  (if (::started? server)
    (-> (http/stop server)
        (assoc ::started? false))
    server))

(defrecord Pedestal []
  component/Lifecycle
  (start [server]
    (start-server server))
  (stop [server]
    (stop-server server)))

(defn server
  "Return a new Pedestal server."
  [config]
  (-> {::http/allowed-origins {:creds true :allowed-origins (constantly true)}
       ::http/join? false
       ::http/port (:bind-port config)
       ::http/resource-path "/public"
       ::http/routes (:routes config)
       ::http/type :jetty
       ::started? false}
      (merge config)
      http/default-interceptors
      (map->Pedestal)))

(defmacro with-server
  "Create a new Pedestal server component, bind the started component
  to `server-sym`, evaluate `body` and stop the component again."
  [[server-sym config] & body]
  `(let [server# (component/start (server ~config))
         ~server-sym server#]
     (try ~@body
          (finally (component/stop server#)))))
