(ns burningswell.nrepl.component
  (:require [clojure.spec.alpha :as s]
            [com.stuartsierra.component :as component]
            [io.pedestal.log :as log]
            [nrepl.server :as nrepl])
  (:import nrepl.server.Server))

(s/def ::bind-address string?)
(s/def ::bind-port (s/nilable int?))
(s/def ::server (s/nilable #(instance? Server %)))

(s/def ::config
  (s/keys :opt-un [::bind-address ::bind-port]))

(s/def ::component
  (s/keys :req-un [::config ::server]))

(def defaults
  "The default NREPL configuration."
  {:bind-address "0.0.0.0"
   :bind-port nil})

(defn start?
  "Returns true if the :bind-port is set, otherwise false."
  [{:keys [config] :as component}]
  (some? (:bind-port config)))

(s/fdef start?
  :args (s/cat :component ::component))

(defn- start-component
  "Start the NREPL `component`."
  [{:keys [config] :as component}]
  (if (start? component)
    (let [server (nrepl/start-server
                  :bind (:bind-address config)
                  :port (:bind-port config))]
      (log/info :msg (format "NREPL component successfully started.") :config config)
      (assoc component :server server))
    component))

(s/fdef start-component
  :args (s/cat :component ::component))

(defn- stop-component
  "Stop the NREPL `component`."
  [{:keys [config server] :as component}]
  (if (start? component)
    (do (some-> server nrepl/stop-server)
        (log/info :msg (format "NREPL component successfully stopped.") :config config)
        (assoc component :server nil))
    component))

(s/fdef stop-component
  :args (s/cat :component ::component))

(defrecord NREPL [config server]
  component/Lifecycle
  (start [component]
    (start-component component))
  (stop [component]
    (stop-component component)))

(defn nrepl
  "Return a new NREPL component."
  [& [opts]]
  (map->NREPL {:config (merge defaults opts)}))

(s/fdef nrepl
  :args (s/cat :opts (s/? ::config)))
