(ns burningswell.kafka.producer
  (:require [burningswell.kafka.core :as core]
            [burningswell.kafka.serializer :as serializer]
            [peripheral.core :refer [defcomponent]])
  (:import [org.apache.kafka.clients.producer KafkaProducer ProducerRecord]))

(defn config
  "Returns the producer config."
  [& [opts]]
  (merge {:bootstrap.servers core/bootstrap-servers} opts))

(defn- make-client
  "Make a `KafkaProducer` client for `config`."
  [config]
  (KafkaProducer.
   (core/properties config)
   (or (:key-serializer config) (serializer/transit))
   (or (:value-serializer config) (serializer/transit))))

(defcomponent Producer [config]
  :client (make-client config) #(.close %))

(defn producer
  "Returns a Kafka producer component."
  [& [opts]]
  (map->Producer {:config (config opts)}))

(defn record-metadata->map [meta]
  {:checksum (.checksum meta)
   :offset (.offset meta)
   :partition (.partition meta)
   :serialized-key-size (.serializedKeySize meta)
   :serialized-value-size (.serializedValueSize meta)
   :timestamp (.timestamp meta)
   :topic (.topic meta)})

(defn flush!
  "Flush all pending messages."
  [producer]
  (.flush (:client producer)))

(defn send!
  "Send a message with `key` and `value` to `topic`."
  [producer topic key value]
  (record-metadata->map
   @(.send (:client producer) (ProducerRecord. topic key value))))
